package com.qkdata.common.config;

import com.qkdata.common.base.enums.SystemResponseEnum;
import com.qkdata.common.base.exception.BusinessException;
import com.qkdata.common.base.model.ResponseData;
import lombok.extern.slf4j.Slf4j;
import org.apache.shiro.authc.AuthenticationException;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.validation.BindingResult;
import org.springframework.web.HttpMediaTypeNotSupportedException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.MissingPathVariableException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.method.annotation.MethodArgumentTypeMismatchException;
import org.springframework.web.servlet.NoHandlerFoundException;

import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;

@RestControllerAdvice
@Slf4j
public class GlobalExceptionHandle {

    @ExceptionHandler(value = MethodArgumentNotValidException.class)
    public ResponseEntity<ResponseData> bindExceptionHandler(MethodArgumentNotValidException e) {
        BindingResult bindingResult = e.getBindingResult();
        String message = bindingResult.getFieldError().getDefaultMessage();
        log.warn("方法参数无效: {}", message);
        return ResponseEntity.badRequest().body(new ResponseData("70000", message));
    }

    @ExceptionHandler(value = HttpRequestMethodNotSupportedException.class)
    public ResponseEntity<ResponseData> httpRequestMethodNotSupportedExceptionHandler(HttpRequestMethodNotSupportedException e) {
        log.warn("不支持的Method: {}", e.getMessage());
        return ResponseEntity.status(HttpStatus.NOT_FOUND).body(new ResponseData(SystemResponseEnum.METHOD_NOT_SUPPORTED));
    }

    @ExceptionHandler(value = Exception.class)
    public ResponseEntity<ResponseData> handler(Exception e) {
        log.error("服务器错误", e);
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(new ResponseData(SystemResponseEnum.ERROR));
    }


    @ExceptionHandler(value = BusinessException.class)
    public ResponseEntity<ResponseData> businessExceptionhandler(BusinessException e) {
        log.warn("业务异常, code: {}, message: {}", e.getCode(), e.getMessage());
        // TODO 从BusinessException获取responseEnum， 不能直接打印e.getMessage()
        return ResponseEntity.status(HttpStatus.OK).body(new ResponseData(e.getCode(), e.getMessage()));
    }

    @ExceptionHandler(value = HttpMessageNotReadableException.class)
    public ResponseEntity<ResponseData> httpMessageConvertExceptionHanlder(HttpMessageNotReadableException e) {
        log.warn("无法解析请求参数: {}", e.getMessage());
        return ResponseEntity.badRequest().body(new ResponseData(SystemResponseEnum.MESSAGE_NOT_READABLE));
    }

    @ExceptionHandler(value = MissingPathVariableException.class)
    public ResponseEntity<ResponseData> missingPathVariableExceptionHanlder(MissingPathVariableException e) {
        log.warn("请求链接缺少参数: {}", e.getMessage());
        return ResponseEntity.badRequest().body(new ResponseData(SystemResponseEnum.MISSING_PATH_VARIABLE));
    }

    @ExceptionHandler(value = MissingServletRequestParameterException.class)
    public ResponseEntity<ResponseData> requestParameterExceptionHandler(MissingServletRequestParameterException e) {
        log.warn("缺少必需的请求参数: {}", e.getMessage());
        return ResponseEntity.badRequest().body(new ResponseData(SystemResponseEnum.MISSING_REQUEST_PARAM));
    }

    @ExceptionHandler(value = MethodArgumentTypeMismatchException.class)
    public ResponseEntity<ResponseData> methodArgumentTypeMismatchExceptionHandler(MethodArgumentTypeMismatchException e) {
        log.warn("参数类别转换失败: {}", e.getMessage());
        return ResponseEntity.badRequest().body(new ResponseData(SystemResponseEnum.ARGUMENT_TYPE_MISMATCH));
    }

    @ExceptionHandler(value = ConstraintViolationException.class)
    public ResponseEntity<ResponseData> constraintViolationExceptionHandler(ConstraintViolationException e) {
        log.warn("参数校验异常: {}", e.getMessage());
        return ResponseEntity.badRequest().body(new ResponseData(SystemResponseEnum.CONSTRAINT_VIOLATION.value(), e.getConstraintViolations().stream().map(ConstraintViolation::getMessage).findFirst().get()));
    }

    @ExceptionHandler(value = HttpMediaTypeNotSupportedException.class)
    public ResponseEntity<ResponseData> httpMediaTypeNotSupportedExceptionHandler(HttpMediaTypeNotSupportedException e) {
        log.warn("不支持MediaType: {}", e.getMessage());
        return ResponseEntity.badRequest().body(new ResponseData(SystemResponseEnum.MEDIA_TYPE_NOT_SUPPORTED));
    }

    @ExceptionHandler(value = NoHandlerFoundException.class)
    public ResponseEntity<ResponseData> noHandlerFoundExceptionHandler(NoHandlerFoundException e) {
        log.warn("no handler found: {}", e.getMessage());
        return ResponseEntity.status(HttpStatus.NOT_FOUND).body(new ResponseData(SystemResponseEnum.NO_HANDLER_NOT_FOUND));
    }


}
