package com.qkdata.sms.service;

import com.qkdata.sms.exception.SmsException;
import com.qkdata.sms.constant.SmsResponseEnum;
import org.springframework.util.StringUtils;

import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public abstract class AbstractPlaceholderResolver implements PlaceholderResolver {

	protected abstract String placeholderKeyword();

	protected abstract Pattern pattern();

	/**
	 * 1. 模板中是否包含占位符， 如果不包含直接返回true
	 * 2. 如果包含， 判断params是否为空， 如果为空，抛出异常
	 * 3. 如果params不为空, 解析template，并且判断params与placeholder是否完全匹配，忽略掉params中多余的参数，如果完全匹配，返回true
	 * @param template
	 * @param params
	 * @return
	 */
	@Override
	public boolean isMatch(String template, Map<String, Object> params) throws SmsException {
		if (!hasPlaceholder(template, placeholderKeyword())) {
			return true;
		}

		if (params == null || params.size() == 0) {
			throw new SmsException("模板中有占位符，但是并未提供参数", SmsResponseEnum.TEMPLATE_PARAMS_MISMATCHED);
		}

		Matcher matcher = pattern().matcher(template);
		while (matcher.find()) {
			String keyword = matcher.group(1);
			String value = (String) params.get(keyword);
			if (StringUtils.isEmpty(value)) {
				throw new SmsException(String.format("模板参数与模板占位符不符, 占位符: %s", keyword),
						SmsResponseEnum.TEMPLATE_PARAMS_MISMATCHED);
			}
		}
		return true;
	}

	@Override
	public String replace(String template, Map<String, Object> params) throws SmsException {
		return template;
	}
}
