package com.qkdata.sms.service;


import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.qkdata.sms.consumer.QueueConstants;
import com.qkdata.sms.exception.SmsException;
import com.qkdata.sms.model.*;
import com.qkdata.sms.constant.SmsResponseEnum;
import com.qkdata.sms.model.*;
import com.qkdata.sms.notification.AliNotifyRequest;
import com.qkdata.sms.notification.LmobileNotifyRequest;
import com.qkdata.sms.notification.NotifyRequest;
import com.qkdata.sms.notification.NotifyTemplate;
import com.qkdata.sms.util.RandomDigitGenerator;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.client.RestTemplate;

import javax.validation.Valid;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import static java.util.stream.Collectors.toList;

@Service("smsService")
@Slf4j
@EnableConfigurationProperties(SmsV1Property.class)
public class SmsService {

	@Autowired
	private RedisService redisService;

	@Autowired
	private TemplateService templateService;

	@Autowired
	private NotifyTemplate notifyTemplate;

	@Autowired
	private RestTemplate restTemplate;

	@Autowired
	private SmsV1Property smsProperty;

	private static final String ALI_SMS_NOTIFY_KEY = ":ali";

	private static final String LMOBILE_SMS_NOTIFY_KEY = ":lmobile";

	private static final String SMS_NOTIFY_LIST_NAME = "sms:notify";

	public void send(SmsMessageCondition condition) throws Exception {

		List<SmsCondition> conditions = condition.getConditions();
		if (!judgeTemplateCodeIsExist(conditions)) {
			throw new SmsException("短信模板不存在", SmsResponseEnum.TEMPLATE_NOT_EXIST);
		}

		List<SmsCondition> captchaSms = new LinkedList<>();
		List<SmsCondition> notificationSms = new LinkedList<>();

		for (SmsCondition smsCondition : conditions) {
			SmsTypeAndChannelDTO smsTypeAndChannelDTO = templateService.getTypeAndChannelByCode(smsCondition.getCode());
			smsCondition.setChannel(smsTypeAndChannelDTO.getChannel());
			if (smsTypeAndChannelDTO.getType() == 1) {
				captchaSms.add(smsCondition);
			}
			if (smsTypeAndChannelDTO.getType() == 2) {
				notificationSms.add(smsCondition);
			}
		}

		if (!CollectionUtils.isEmpty(captchaSms)) {
			redisService.lpush(QueueConstants.CAPTCHA_QUEUE_NAME, jsonFrom(captchaSms));
		}

		if (!CollectionUtils.isEmpty(notificationSms)) {
			redisService.lpush(QueueConstants.NOTIFICATION_QUEUE_NAME, jsonFrom(notificationSms));
		}
	}

	private List<String> jsonFrom(List<SmsCondition> smses) {
		if (CollectionUtils.isEmpty(smses)) {
			return Collections.emptyList();
		}
		ObjectMapper objectMapper = new ObjectMapper();
		return smses.stream().map(sms -> {
			try {
				return objectMapper.writeValueAsString(sms);
			} catch (JsonProcessingException e) {
				// TODO
				e.printStackTrace();
			}
			return null;
		}).collect(toList());
	}

	private Boolean judgeTemplateCodeIsExist(List<SmsCondition> smsConditionList) {
		List<String> codeList = new ArrayList<>();
		for (SmsCondition smsCondition : smsConditionList) {
			if (!codeList.contains(smsCondition.getCode())) {
				codeList.add(smsCondition.getCode());
			}
		}
		Integer count = templateService.getTemplateCount(codeList);
		if (count.equals(codeList.size())) {
			return true;
		}
		return false;
	}

	public void aliNotify(List<SmsReport> smsReportList) {
		for (SmsReport smsReport : smsReportList) {
			log.info("阿里回调参数: {}", smsReport);
			if (StringUtils.isEmpty(smsReport.getOut_id())) {
				continue;
			}
			String notifyUrlKey = smsReport.getOut_id() + ALI_SMS_NOTIFY_KEY;
			String notifyUrl = notifyTemplate.getNotifyUrl(notifyUrlKey);
			if (StringUtils.isEmpty(notifyUrl)) {
				continue;
			}

			String notifyString = objectToJsonStirng(smsReport);
			NotifyRequest request = new AliNotifyRequest(smsReport);
			notifyTemplate.notify(SMS_NOTIFY_LIST_NAME, notifyString, notifyUrl, notifyUrlKey, request);
		}
	}

	public void lmobileNotify(LmobileNotifyCondition lmobileNotifyCondition) {
		log.info("lmobile回调通知参数: {}", lmobileNotifyCondition);
		if (StringUtils.isEmpty(lmobileNotifyCondition.getClientMsgId())) {
			return;
		}
		String notifyUrlKey = lmobileNotifyCondition.getClientMsgId() + LMOBILE_SMS_NOTIFY_KEY;
		String notifyUrl = notifyTemplate.getNotifyUrl(notifyUrlKey);
		if (StringUtils.isEmpty(notifyUrl)) {
			return;
		}
		String notifyString = objectToJsonStirng(lmobileNotifyCondition);
		NotifyRequest request = new LmobileNotifyRequest(lmobileNotifyCondition);

		notifyTemplate.notify(SMS_NOTIFY_LIST_NAME, notifyString, notifyUrl, notifyUrlKey, request);
	}

	// TODO 抽取
	private String objectToJsonStirng(Object object) {
		ObjectMapper objectMapper = new ObjectMapper();
		String result = "";
		try {
			result = objectMapper.writeValueAsString(object);
		} catch (JsonProcessingException e) {
			log.error("smsCondition序列化失败", e);
		}
		return result;
	}

	public void sendSms(@Valid SmsV1Condition smsCondition) throws SmsException, IOException {
		log.info("手机号：" + smsCondition.getMobile() + "；内容：" + smsCondition.getMessageContent());
		String url = String.join("", smsProperty.getUrl(), "?format=json&data={json}");

		String response = restTemplate.getForObject(url, String.class, requestJson(smsCondition.getMobile(), smsCondition.getMessageContent()));
		log.info("发送短信响应结果: " + response);
		ObjectMapper objectMapper = new ObjectMapper();
		SmsApiV1Response smsApiResponse = objectMapper.readValue(response, SmsApiV1Response.class);

		if (SmsApiV1Response.SUCCESS_RESPONSE_KEY.equals(smsApiResponse.getStatus())) {
			return;
		} else {
			throw new SmsException(String.format("发送短信失败, 失败原因: %s", SmsApiV1Response.RESPONSE_RESULT.get(smsApiResponse.getMsg())), SmsResponseEnum.SEND_ERROR);
		}

	}

	private String requestJson(String mobile, String messageContent) throws SmsException {

		try {
			SmsApiV1Condition condition = new SmsApiV1Condition();
			condition.setApikey(smsProperty.getApikey());
			if (messageContent == null || messageContent.trim().equals(""))
				condition.setContent(URLEncoder.encode(content(mobile), "UTF-8"));
			else
				condition.setContent(URLEncoder.encode(messageContent, "UTF-8"));
			condition.setMobile(mobile);
			condition.setUsername(smsProperty.getUsername());
			String md5Password = bytesToHex(MessageDigest.getInstance("MD5").digest(smsProperty.getPassword().getBytes("UTF-8")));
			condition.setPassword_md5(md5Password);
			ObjectMapper objectMapper = new ObjectMapper();
			return objectMapper.writeValueAsString(condition);
		} catch (UnsupportedEncodingException | NoSuchAlgorithmException | JsonProcessingException e) {
			log.error("发送短信失败", e);
			throw new SmsException("发送短信失败", SmsResponseEnum.SEND_ERROR);
		}
	}

	private String content(String mobile) {
		String captcha = RandomDigitGenerator.generate();
		return String.format(smsProperty.getTemplate(), captcha);
	}

	/**
	 * 二进制转十六进制
	 *
	 * @param bytes
	 * @return
	 */
	public static String bytesToHex(byte[] bytes) {
		StringBuilder md5str = new StringBuilder();
		//把数组每一字节换成16进制连成md5字符串
		int digital;
		for (int i = 0; i < bytes.length; i++) {
			digital = bytes[i];

			if (digital < 0) {
				digital += 256;
			}
			if (digital < 16) {
				md5str.append("0");
			}
			md5str.append(Integer.toHexString(digital));
		}
		return md5str.toString().toLowerCase();
	}
}
