package com.qkdata.sms.service;

import com.qkdata.sms.constant.SmsTemplate;
import com.qkdata.sms.exception.SmsException;
import com.qkdata.sms.exception.SmsRemoteApiException;
import com.qkdata.sms.lmobile.RegularSmsRequest;
import com.qkdata.sms.model.LmobileResponse;
import com.qkdata.sms.model.SmsCondition;
import com.qkdata.sms.constant.SmsResponseEnum;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;
import org.springframework.web.client.RestTemplate;

import java.util.Map;

@Service
@Slf4j
public class LmobileSmsSender implements SmsSender {

	@Autowired
	private TemplateService templateService;

	@Autowired
	private RestTemplate restTemplate;

	@Autowired
	private RedisService redisService;

	@Value("${lmobile.user}")
	private String user;

	@Value("${lmobile.password}")
	private String password;

	@Value("${lmobile.productId}")
	private String productId;

	@Value("${lmobile.url}")
	private String url;

	private static final String LMOBILE_TYPE = ":lmobile";

	@Override
	public void send(SmsCondition sms) throws SmsException, SmsRemoteApiException {
		SmsTemplate smsTemplate = templateService.getTemplateByCode(sms.getCode());
		Map<String, Object> paramsMap = sms.getParams();
		String content = smsTemplate.getLmobileTemplateContent();

		PlaceholderResolver placeholderResolver = new LmobilePlaceholderResolver();
		content = placeholderResolver.replace(content, paramsMap);
		content = appendSignNameToContent(content, smsTemplate.getLmobileSignName());

		RegularSmsRequest regularSmsRequest = new RegularSmsRequest();
		regularSmsRequest.setUser(user);
		regularSmsRequest.setPassword(password);
		regularSmsRequest.setProductId(productId);
		regularSmsRequest.setMobiles(sms.getMobile());
		if (!StringUtils.isEmpty(sms.getOutId())) {
			regularSmsRequest.setKey(sms.getOutId());
		}
		regularSmsRequest.setContent(content);
		try {
			LmobileResponse lmobileResponse = restTemplate.postForObject(url, regularSmsRequest, LmobileResponse.class);
			if (!lmobileResponse.getState().equals(0)) {
				if (lmobileResponse.getState().equals(1025)) {
					log.error("Lmobile短信平台余额不足:{}", lmobileResponse.getMsgState());
				} else {
					log.error("Lmobile短信发送失败:{}", lmobileResponse.getMsgState());
				}
				throw new SmsRemoteApiException("短信发送失败", SmsResponseEnum.SEND_ERROR);
			} else {
				log.info("{}---发送成功:{}", sms.getMobile(), content);
				if (!StringUtils.isEmpty(sms.getOutId()) && !StringUtils.isEmpty(sms.getNotifyUrl())) {
					redisService.set(sms.getOutId() + LMOBILE_TYPE, sms.getNotifyUrl());
				}
			}
		} catch (Exception e) {
			log.error("lmobie发送失败", e);
			throw new SmsRemoteApiException("短信发送失败", SmsResponseEnum.SEND_ERROR);
		}

	}

	private String appendSignNameToContent(String content, String signName) {
		return content + "【" + signName + "】";
	}

}
