package com.qkdata.sms.service;

import com.aliyuncs.DefaultAcsClient;
import com.aliyuncs.IAcsClient;
import com.aliyuncs.dysmsapi.model.v20170525.SendSmsRequest;
import com.aliyuncs.dysmsapi.model.v20170525.SendSmsResponse;
import com.aliyuncs.exceptions.ClientException;
import com.aliyuncs.http.MethodType;
import com.aliyuncs.profile.DefaultProfile;
import com.aliyuncs.profile.IClientProfile;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.qkdata.sms.constant.SmsTemplate;
import com.qkdata.sms.exception.SmsException;
import com.qkdata.sms.constant.SmsResponseEnum;
import com.qkdata.sms.exception.SmsRemoteApiException;
import com.qkdata.sms.model.SmsCondition;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import java.util.Map;

@Service
@Slf4j
public class AliSmsSender implements SmsSender {

	@Autowired
	private TemplateService templateService;

	@Autowired
	private RedisService redisService;

	@Value("${aliyun.accessKeyId}")
	private String accessKeyId;

	@Value("${aliyun.accessKeySecret}")
	private String accessKeySecret;

	private static final String ALI_TYPE = ":ali";

	private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();

	@Override
	public void send(SmsCondition sms) throws SmsException, SmsRemoteApiException {
		SmsTemplate smsTemplate = templateService.getTemplateByCode(sms.getCode());
		Map<String, Object> paramsMap = sms.getParams();
		String content = smsTemplate.getAlibabaTemplateContent();

		PlaceholderResolver placeholderResolver = new AliPlaceholderResolver();
		if (!placeholderResolver.isMatch(content, paramsMap)) {
			return;
		}
		try {
			SendSmsResponse sendSmsResponse = submitSms(sms.getMobile(), smsTemplate.getAlibabaSignName(),
					smsTemplate.getAlibabaTemplateCode(), paramsToJson(paramsMap), sms.getOutId(), accessKeyId,
					accessKeySecret);
			if (sendSmsResponse.getCode() != null && !"OK".equals(sendSmsResponse.getCode())) {
				if (sendSmsResponse.getCode().equals("isv.AMOUNT_NOT_ENOUGH")) {
					log.error("阿里短信平台余额不足：{},{}", sendSmsResponse.getCode(), sendSmsResponse.getMessage());
				} else {
					log.error("阿里短信发送失败：{},{}", sendSmsResponse.getCode(), sendSmsResponse.getMessage());
				}
				throw new SmsRemoteApiException("阿里短信发送失败", SmsResponseEnum.SEND_ERROR);
			} else {
				log.info("{}---发送成功:{}", sms.getMobile(), content);
				if (!StringUtils.isEmpty(sms.getOutId()) && !StringUtils.isEmpty(sms.getNotifyUrl())) {
					redisService.set(sms.getOutId() + ALI_TYPE, sms.getNotifyUrl());
				}
			}
		} catch (Exception e) {
			log.error("调用阿里云短信平台失败", e);
			throw new SmsRemoteApiException("阿里短信发送失败", SmsResponseEnum.SEND_ERROR);
		}


	}

	private String paramsToJson(Map<String, Object> paramsMap) {
		String paramsJson = "";
		if (paramsMap == null || paramsMap.size() == 0) {
			return paramsJson;
		}
		try {
			paramsJson = OBJECT_MAPPER.writeValueAsString(paramsMap);
		} catch (JsonProcessingException e) {
			// do nothing
		}
		return paramsJson;
	}

	private SendSmsResponse submitSms(String mobile, String signName, String templateCode, String templateParams,
			String outId, String accessKeyId, String accessKeySecret) throws ClientException {
		System.setProperty("sun.net.client.defaultConnectTimeout", "10000");
		System.setProperty("sun.net.client.defaultReadTimeout", "10000");
//		final String product = "Dysmsapi";//短信API产品名称（短信产品名固定，无需修改）
//		final String domain = "dysmsapi.aliyuncs.com";//短信API产品域名（接口地址固定，无需修改）
		//初始化ascClient,暂时不支持多region（请勿修改）
		IClientProfile profile = DefaultProfile.getProfile("cn-hangzhou", accessKeyId, accessKeySecret);
		//        DefaultProfile.addEndpoint("cn-hangzhou", "cn-hangzhou", product, domain);
		IAcsClient acsClient = new DefaultAcsClient(profile);
		SendSmsRequest request = new SendSmsRequest();
		request.setMethod(MethodType.POST);
		//必填:待发送手机号。支持以逗号分隔的形式进行批量调用，批量上限为1000个手机号码,批量调用相对于单条调用及时性稍有延迟,验证码类型的短信推荐使用单条调用的方式；发送国际/港澳台消息时，接收号码格式为00+国际区号+号码，如“0085200000000”
		request.setPhoneNumbers(mobile);
		//必填:短信签名-可在短信控制台中找到
		request.setSignName(signName);
		//必填:短信模板-可在短信控制台中找到，发送国际/港澳台消息时，请使用国际/港澳台短信模版
		request.setTemplateCode(templateCode);
		//可选:模板中的变量替换JSON串,如模板内容为"亲爱的${name},您的验证码为${code}"时,此处的值为
		//友情提示:如果JSON中需要带换行符,请参照标准的JSON协议对换行符的要求,比如短信内容中包含\r\n的情况在JSON中需要表示成\\r\\n,否则会导致JSON在服务端解析失败
		request.setTemplateParam(templateParams);
		//可选:outId为提供给业务方扩展字段,最终在短信回执消息中将此值带回给调用者
		if (!StringUtils.isEmpty(outId)) {
			request.setOutId(outId);
		}

		return acsClient.getAcsResponse(request);
	}


}
