package com.qkdata.sms.exception;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;

import com.qkdata.sms.constant.ConstantResponseEnum;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.validation.BindingResult;
import org.springframework.web.HttpMediaTypeNotSupportedException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.MissingPathVariableException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;

import com.qkdata.sms.constant.ResponseData;

import lombok.extern.slf4j.Slf4j;
import org.springframework.web.method.annotation.MethodArgumentTypeMismatchException;
import org.springframework.web.servlet.NoHandlerFoundException;

/**
 * 业务异常处理类
 * @author wangcy
 *
 */
@RestControllerAdvice
@Slf4j
public class ControllerExceptionHandleAdvice {

    @ExceptionHandler(value = MethodArgumentNotValidException.class)
    public ResponseEntity<?> bindExceptionHandler(HttpServletRequest req, HttpServletResponse res, MethodArgumentNotValidException e) {
        BindingResult bindingResult = e.getBindingResult();
        String message = bindingResult.getFieldError().getDefaultMessage();
        log.info(message);
        return ResponseEntity.badRequest().body(new ResponseData(HttpStatus.BAD_REQUEST, message));
    }

    @ExceptionHandler(value = BusinessException.class)
    public ResponseEntity<?> businessExceptionHandler(HttpServletRequest req, HttpServletResponse res, BusinessException e) {
        String message = e.getMessage();
        log.info(message);
        return ResponseEntity.badRequest().body(new ResponseData(HttpStatus.INTERNAL_SERVER_ERROR, message));
    }

    @ExceptionHandler(HttpMessageNotReadableException.class)
    public ResponseEntity<?> httpMessageConvertExceptionHanlder(HttpMessageNotReadableException e) {
        return responseData(ConstantResponseEnum.JSON_PARSE_ERROR, e);
    }

    @ExceptionHandler(MissingPathVariableException.class)
    public ResponseEntity<?> missingPathVariableExceptionHanlder(MissingPathVariableException e) {
        return responseData(ConstantResponseEnum.MISSING_PATH_VARIABLE, e);
    }

    @ExceptionHandler(MissingServletRequestParameterException.class)
    public ResponseEntity<?> requestParameterExceptionHandler(MissingServletRequestParameterException e) {
        return responseData(ConstantResponseEnum.MISSING_REQUEST_PARAMETER, e);
    }

    @ExceptionHandler(MethodArgumentTypeMismatchException.class)
    public ResponseEntity<?> methodArgumentTypeMismatchExceptionHandler(MethodArgumentTypeMismatchException e) {
        return responseData(ConstantResponseEnum.METHOD_ARGUMENT_TYPE_MIS_MATCH, e);
    }

    @ExceptionHandler(ConstraintViolationException.class)
    public ResponseEntity<?> constraintViolationExceptionHandler(ConstraintViolationException e) {
        log(e);
        return ResponseEntity.badRequest().body(ResponseData.builder()
                .code(ConstantResponseEnum.CONSTRAINT_VIOLATION.value())
                .message(e.getConstraintViolations().stream().map(ConstraintViolation::getMessage).findFirst().get())
                .build());
    }

    @ExceptionHandler(HttpMediaTypeNotSupportedException.class)
    public ResponseEntity<?> httpMediaTypeNotSupportedExceptionHandler(HttpMediaTypeNotSupportedException e) {
        return responseData(ConstantResponseEnum.MEDIA_TYPE_NOT_SUPPORTED, e);
    }


    @ExceptionHandler(HttpRequestMethodNotSupportedException.class)
    public ResponseEntity<?> httpRequestMethodNotSupportedExceptionHandler(HttpRequestMethodNotSupportedException e) {
        return responseData(ConstantResponseEnum.REQUEST_METHOD_NOT_SUPPORTED, e);
    }

    @ExceptionHandler(NoHandlerFoundException.class)
    public ResponseEntity<?> noHandlerFoundExceptionHandler(NoHandlerFoundException e) {
        return responseData(ConstantResponseEnum.NO_HANDLER_FOUND, e);
    }

    private void log(Throwable e) {
        log.error(e.getMessage(), e);
    }

    private ResponseEntity<?> responseData(ConstantResponseEnum responseEnum, Throwable e) {
        log(e);
        return ResponseEntity.badRequest().body(ResponseData.builder()
                .result(responseEnum)
                .build());
    }

}
