package com.qkdata.sms.consumer;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.github.rholder.retry.*;
import com.google.common.base.Predicates;
import com.qkdata.sms.service.RedisService;
import com.qkdata.sms.exception.SmsRemoteApiException;
import com.qkdata.sms.model.SmsCondition;
import com.qkdata.sms.service.SmsSender;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.DependsOn;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

@Component
@DependsOn
@Slf4j
public class TaskConsumer {

	@Autowired
	private RedisConnectionFactory jedisConnectionFactory;

	@Autowired
	@Qualifier("aliSmsSender")
	private SmsSender aliSmsSender;

	@Autowired
	@Qualifier("lmobileSmsSender")
	private SmsSender lmobileSmsSender;

	@Value("${retry.count}")
	private Integer retryCount;

	@Autowired
	private RedisService redisService;

	private static final String ERROR_LIST = "sms:error";

	private static final ThreadPoolExecutor THREAD_POOL_EXECUTOR = new ThreadPoolExecutor(2,
			Runtime.getRuntime().availableProcessors() * 2, 10, TimeUnit.SECONDS, new LinkedBlockingQueue<>());


	@PostConstruct
	public void init() {
		THREAD_POOL_EXECUTOR.execute(() -> {
			RedisConnection connection = jedisConnectionFactory.getConnection();
			ObjectMapper objectMapper = new ObjectMapper();
			Retryer<Boolean> retryer = buildRetryer();
			while (true) {
				List<byte[]> bytes = connection.listCommands()
						.bLPop(0, QueueConstants.CAPTCHA_QUEUE_NAME.getBytes(StandardCharsets.UTF_8),
								QueueConstants.NOTIFICATION_QUEUE_NAME.getBytes(StandardCharsets.UTF_8));

				bytes.forEach(task -> {
					String value = new String(task);
					// 返回结果中第一个元素为队列名称本身， 第二个元素是短信内容, 故忽略
					if (QueueConstants.CAPTCHA_QUEUE_NAME.equals(value) || QueueConstants.NOTIFICATION_QUEUE_NAME
							.equals(value)) {
						return;
					}
					SmsCondition smsCondition = deserializeSms(objectMapper, value);
					if (smsCondition == null) {
						return;
					}
					retry(retryer, new SendSmsTask(smsCondition, aliSmsSender, lmobileSmsSender));
				});
			}
		});
	}

	private SmsCondition deserializeSms(ObjectMapper objectMapper, String sms) {
		SmsCondition smsCondition;
		try {
			smsCondition = objectMapper.readValue(sms, SmsCondition.class);
		} catch (IOException e) {
			log.error("反序列化短信参数失败", e);
			return null;
		}
		return smsCondition;
	}

	private Retryer<Boolean> buildRetryer() {
		return RetryerBuilder.<Boolean>newBuilder().retryIfExceptionOfType(SmsRemoteApiException.class)
				.retryIfResult(Predicates.equalTo(false))
				//                .withWaitStrategy(WaitStrategies.fixedWait(5, TimeUnit.SECONDS))
				.withWaitStrategy(WaitStrategies.exponentialWait(1000, 60, TimeUnit.SECONDS))
				//                .withWaitStrategy(WaitStrategies.fibonacciWait(1000, 10, TimeUnit.SECONDS))
				.withStopStrategy(StopStrategies.stopAfterAttempt(retryCount)).withRetryListener(new SmsRetryListener())
				.build();
	}

	private void retry(Retryer<Boolean> retryer, SendSmsTask task) {
		try {
			// TODO 重试Listener； 重试失败后， 放入另外队列中
			retryer.call(task);
		} catch (ExecutionException | RetryException e) {
			log.error("重试发送失败", e);
			String smsCondition = objectToJsonStirng(task.getSmsCondition());
			redisService.rpush(ERROR_LIST, smsCondition);

		}
	}

	private String objectToJsonStirng(Object object) {
		ObjectMapper objectMapper = new ObjectMapper();
		String result = "";
		try {
			result = objectMapper.writeValueAsString(object);
		} catch (JsonProcessingException e) {
			log.error("smsCondition序列化失败", e);
		}
		return result;
	}
}
