package com.qkdata.biz.sys.service;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.RandomUtil;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.google.common.collect.Lists;
import com.qkdata.biz.enums.AccountStatusEnum;
import com.qkdata.biz.enums.AccountTypeEnum;
import com.qkdata.biz.enums.ProductTypeEnum;
import com.qkdata.biz.management.entity.OrgSurplusPO;
import com.qkdata.biz.management.entity.OrgUserReceiveRecordPO;
import com.qkdata.biz.management.entity.UserCourseAuthPO;
import com.qkdata.biz.management.service.OrgSurplusService;
import com.qkdata.biz.management.service.OrgUserReceiveRecordService;
import com.qkdata.biz.management.service.UserCourseAuthService;
import com.qkdata.biz.sys.entity.SysUserExtPO;
import com.qkdata.biz.sys.entity.SysUserPO;
import com.qkdata.biz.sys.mapper.SysUserExtMapper;
import com.qkdata.biz.sys.mapper.SysUserMapper;
import com.qkdata.biz.sys.vo.QueryUserModel;
import com.qkdata.biz.sys.vo.SysRoleModel;
import com.qkdata.biz.sys.vo.SysUserListModel;
import com.qkdata.biz.sys.vo.SysUserModel;
import com.qkdata.biz.web.vo.FullUserInfo;
import com.qkdata.biz.web.vo.ModifyUserInfoModel;
import com.qkdata.common.base.enums.CodeEnum;
import com.qkdata.common.base.exception.BusinessException;
import com.qkdata.common.base.model.PageResult;
import com.qkdata.common.base.service.impl.BaseServiceImpl;
import com.qkdata.common.util.UserContext;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.shiro.crypto.hash.Sha256Hash;
import org.apache.tomcat.jni.Local;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;

import java.time.LocalDateTime;
import java.util.List;

@Service
public class SysUserService extends BaseServiceImpl<SysUserMapper, SysUserPO> {
    @Autowired
    private SysUserRoleService sysUserRoleService;
    @Autowired
    private SysRoleService sysRoleService;
    @Autowired
    private SysUserExtMapper sysUserExtMapper;
    @Autowired
    private OrgSurplusService orgSurplusService;
    @Autowired
    private OrgUserReceiveRecordService orgUserReceiveRecordService;
    @Autowired
    private UserCourseAuthService userCourseAuthService;

    public SysUserPO getByUsername(String username) {
        return baseMapper.selectOne(Wrappers.<SysUserPO>lambdaQuery().eq(SysUserPO::getUsername,username));
    }

    public List<String> queryAllPerms(Long userId) {
        return baseMapper.queryAllPerms(userId);
    }

    public PageResult<SysUserListModel> queryPageList(QueryUserModel queryUserModel) {
        Page<SysUserListModel> page = new Page<>(queryUserModel.getPageIndex(),queryUserModel.getPageSize());
        List<SysUserListModel> pageList = baseMapper.queryPageList(page,queryUserModel);
        for (SysUserListModel sysUser : pageList){
            List<SysRoleModel> roles = sysRoleService.getUserRoles(sysUser.getId());
            sysUser.setRoleIdList(roles);
        }
        return PageResult.<SysUserListModel>builder().code(CodeEnum.SUCCESS.getCode()).count(page.getTotal()).data(pageList).build();
    }

    public List<Long> queryRoleIdList(Long userId) {
        return sysUserRoleService.queryRoleIdList(userId);
    }

    @Transactional
    public Long saveUser(SysUserModel sysUserModel) {
        SysUserPO po = getByUsername(sysUserModel.getUsername());
        if (po != null){
            throw new BusinessException("帐号已存在");
        }
        po = new SysUserPO();
        BeanUtils.copyProperties(sysUserModel,po);
        po.setSalt(RandomStringUtils.randomAlphanumeric(20));
        po.setPassword(new Sha256Hash(sysUserModel.getPassword(), po.getSalt()).toHex());
        po.setLastLoginTime(LocalDateTime.now());
        po.setStatus(sysUserModel.getStatus() != null ? sysUserModel.getStatus() : AccountStatusEnum.UNACTIVATE);
        save(po);
        List<Long> roleIds = null;
        if (CollUtil.isNotEmpty(sysUserModel.getRoleIdList())){
            roleIds = sysUserModel.getRoleIdList();
        }else {
            //默认为运营人员
            roleIds = Lists.newArrayList(2L);
        }
        sysUserRoleService.saveOrUpdateUserRole(po.getId(),roleIds);
        return po.getId();
    }

    @Transactional
    public Long updateUser(SysUserModel sysUserModel) {
        SysUserPO po = getById(sysUserModel.getId());
        if (po == null){
            throw new BusinessException("请求错误，用户不存在");
        }
        po.setStatus(sysUserModel.getStatus());
        po.setNickName(sysUserModel.getNickName());
        updateById(po);
        return po.getId();
//        sysUserRoleService.saveOrUpdateUserRole(po.getId(),sysUserModel.getRoleIdList());
    }

    public void updatePassword(SysUserPO user, String password, String newPassword) {
        //sha256加密
        String pwd = new Sha256Hash(password, UserContext.getUser().getSalt()).toHex();
        if (!user.getPassword().equals(pwd)){
            throw new BusinessException("原密码错误");
        }
        //sha256加密
        String newPwd = new Sha256Hash(newPassword, UserContext.getUser().getSalt()).toHex();
        user.setPassword(newPwd);
        updateById(user);
    }

    @Transactional
    public void removeUsers(List<Long> ids) {
        removeByIds(ids);
        sysUserRoleService.deleteByUserIds(ids);
    }

    public FullUserInfo findFullUserInfo(String username) {
        return baseMapper.selectFullUserInfo(username);
    }

    public void registerUser(String mobile) {
        SysUserPO userPo = new SysUserPO();
        userPo.setUsername(mobile);
        userPo.setStatus(AccountStatusEnum.ENABLE);
        userPo.setActivateTime(LocalDateTime.now());
        userPo.setNickName("用户"+ RandomUtil.randomNumbers(4));
        save(userPo);
        List<Long> roleIds = Lists.newArrayList(5L);
        sysUserRoleService.saveOrUpdateUserRole(userPo.getId(),roleIds);
        SysUserExtPO extUser = new SysUserExtPO();
        extUser.setUserId(userPo.getId());
        extUser.setType(AccountTypeEnum.USER);
        sysUserExtMapper.insert(extUser);
    }
    public Long getCurrentUserEnterpriseId(){
        String username = UserContext.getUser().getUsername();
        FullUserInfo userInfo = findFullUserInfo(username);
        Long enterpriesId = userInfo.getId();
        return enterpriesId;
    }

    /**
     * 更新用户VIP期限
     * 用户为普通用户时：更新开始结束时间，更新用户为会员
     * 用户为会员或企业会员时：判断用户VIP是否过期，已过期：更新开始、结束时间，未过期：延长结束时间
     * @param userId
     * @param startTime
     * @param month 期限（几人月）
     */
    public void updateVIPTimeLimit(Long userId, LocalDateTime startTime, int month) {
        SysUserExtPO userExtPO = sysUserExtMapper.selectOne(Wrappers.<SysUserExtPO>lambdaQuery().eq(SysUserExtPO::getUserId,userId));
        if (userExtPO != null){
            if (userExtPO.getType() == AccountTypeEnum.USER){
                userExtPO.setVipStartTime(startTime);
                userExtPO.setVipEndTime(startTime.plusMonths(month));
                userExtPO.setType(AccountTypeEnum.VIP);
            }else {
                LocalDateTime now = LocalDateTime.now();
                if (now.isAfter(userExtPO.getVipStartTime()) && now.isBefore(userExtPO.getVipEndTime())){
                    userExtPO.setVipEndTime(userExtPO.getVipEndTime().plusMonths(month));
                }else {
                    userExtPO.setVipStartTime(startTime);
                    userExtPO.setVipEndTime(startTime.plusMonths(month));
                }
            }
            sysUserExtMapper.updateById(userExtPO);
        }
    }

    public Long getUserEnterpriesId(Long userId) {
        SysUserExtPO userExtPO = sysUserExtMapper.selectOne(Wrappers.<SysUserExtPO>lambdaQuery().eq(SysUserExtPO::getUserId,userId));
        if (userExtPO != null){
            return userExtPO.getEnterpriseId();
        }
        return null;
    }

    /**
     * 修改用户个人信息
     * @param model
     */
    public void modifyUserInfo(ModifyUserInfoModel model) {
        SysUserPO userPO = getById(model.getId());
        if (userPO != null){
            userPO.setNickName(model.getNickName());
            updateById(userPO);
        }
        SysUserExtPO extPO = sysUserExtMapper.selectOne(Wrappers.<SysUserExtPO>lambdaQuery().eq(SysUserExtPO::getUserId,model.getId()));
        if(extPO != null){
            extPO.setName(model.getName());
            extPO.setSex(model.getSex());
            extPO.setAge(model.getAge());
            extPO.setAvatarUrl(model.getAvatarUrl());
            sysUserExtMapper.updateById(extPO);
        }
    }

    /**
     * 当前用户领取企业VIP
     */
    @Transactional
    public void receiveOrgVIP() {
        FullUserInfo userInfo = findFullUserInfo(UserContext.getUser().getUsername());
        if (userInfo.getType() != AccountTypeEnum.ENTERPRISE_VIP || userInfo.getEnterpriseId() == null){
            throw new BusinessException("只有企业成员才可领取会员");
        }
        if (userInfo.getVipStartTime() != null && userInfo.getVipEndTime() != null){
            LocalDateTime now = LocalDateTime.now();
            if (now.isAfter(userInfo.getVipStartTime()) && now.isBefore(userInfo.getVipEndTime())){
                throw new BusinessException("您的企业会员生效中，请过期后再领取");
            }
        }
        //减少企业余额
        OrgSurplusPO po = new OrgSurplusPO();
        po.setOrgId(userInfo.getEnterpriseId());
        po.setProductType(ProductTypeEnum.ENTERPRISE_VIP);
        po.setCount(1);
        orgSurplusService.subtractCount(po);
        //添加领用记录
        OrgUserReceiveRecordPO orgUserReceiveRecordPO = new OrgUserReceiveRecordPO();
        orgUserReceiveRecordPO.setUserId(userInfo.getId());
        orgUserReceiveRecordPO.setOrgId(userInfo.getEnterpriseId());
        orgUserReceiveRecordPO.setCount(1);
        orgUserReceiveRecordPO.setProductType(ProductTypeEnum.ENTERPRISE_VIP);
        orgUserReceiveRecordService.save(orgUserReceiveRecordPO);


    }

    /**
     * 领取企业课程
     * @param courseId
     */
    @Transactional
    public void receiveOrgCourse(Long courseId) {
        FullUserInfo userInfo = findFullUserInfo(UserContext.getUser().getUsername());
        if (userInfo.getType() != AccountTypeEnum.ENTERPRISE_VIP || userInfo.getEnterpriseId() == null){
            throw new BusinessException("只有企业成员才可领取会员");
        }
        //验证当前用户是否领取过该课程，并且未过期，则提示不能领取
        UserCourseAuthPO auth =  userCourseAuthService.getByUserIdAndCourseId(userInfo.getId(),courseId);
        if (auth != null){
            LocalDateTime now = LocalDateTime.now();
            if (now.isAfter(auth.getStartTime()) && now.isBefore(auth.getEndTime())){
                throw new BusinessException("您已领取过该课程，并在有效期内，不可重覆领取");
            }
        }
        //减少企业余额
        OrgSurplusPO po = new OrgSurplusPO();
        po.setOrgId(userInfo.getEnterpriseId());
        po.setCourseId(courseId);
        po.setProductType(ProductTypeEnum.ENTERPRISE_COURSE_BUY);
        po.setCount(1);
        orgSurplusService.subtractCount(po);
        //添加领用记录
        OrgUserReceiveRecordPO orgUserReceiveRecordPO = new OrgUserReceiveRecordPO();
        orgUserReceiveRecordPO.setUserId(userInfo.getId());
        orgUserReceiveRecordPO.setOrgId(userInfo.getEnterpriseId());
        orgUserReceiveRecordPO.setCourseId(courseId);
        orgUserReceiveRecordPO.setCount(1);
        orgUserReceiveRecordPO.setProductType(ProductTypeEnum.ENTERPRISE_COURSE_BUY);
        orgUserReceiveRecordService.save(orgUserReceiveRecordPO);
    }
}
