package com.qkdata.biz.management.controller;


import com.google.common.base.Function;
import com.google.common.collect.Lists;
import com.qkdata.biz.common.BizConstants;
import com.qkdata.biz.enums.OrgTypeEnum;
import com.qkdata.biz.management.entity.OrganizationPO;
import com.qkdata.biz.management.service.OrganizationService;
import com.qkdata.biz.management.vo.InstitutionModel;
import com.qkdata.biz.management.vo.QueryOrgModel;
import com.qkdata.common.annotation.SysLog;
import com.qkdata.common.base.enums.CodeEnum;
import com.qkdata.common.base.exception.BusinessException;
import com.qkdata.common.base.model.PageResult;
import com.qkdata.common.base.model.Result;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.apache.shiro.authz.annotation.Logical;
import org.apache.shiro.authz.annotation.RequiresRoles;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.Arrays;
import java.util.List;

/**
 * <p>
 * 培训机构 前端控制器
 * </p>
 *
 * @author liuyang
 * @since 2021-04-27
 */
@Api(tags = "培训机构管理")
@RestController
@RequestMapping("/api/mgr/institution")
public class InstitutionController {

    @Autowired
    private OrganizationService orgService;

    @ApiOperation("添加机构信息")
    @PostMapping("/save")
    @SysLog(value = "添加培训机构")
    @RequiresRoles(value = {BizConstants.ROLE_ADMIN,BizConstants.ROLE_OPERATOR},logical = Logical.OR)
    public Result<String> save(@RequestBody @Validated InstitutionModel model){
        OrganizationPO orgPo = new OrganizationPO();
        BeanUtils.copyProperties(model,orgPo);
        orgPo.setType(OrgTypeEnum.INSTITUTION);
        orgService.save(orgPo);
        return Result.succeed("ok");
    }
    @ApiOperation("修改机构信息")
    @PostMapping("/update")
    @SysLog(value = "修改机构信息")
    @RequiresRoles(value = {BizConstants.ROLE_ADMIN,BizConstants.ROLE_OPERATOR},logical = Logical.OR)
    public Result<String> update(@RequestBody @Validated InstitutionModel model){
        OrganizationPO po = new OrganizationPO();
        BeanUtils.copyProperties(model,po);
        orgService.updateById(po);
        return Result.succeed("ok");
    }

    @ApiOperation("删除机构信息")
    @PostMapping("/delete")
    @SysLog(value = "删除机构信息")
    @RequiresRoles(value = {BizConstants.ROLE_ADMIN,BizConstants.ROLE_OPERATOR},logical = Logical.OR)
    public Result<String> delete(@RequestBody Long[] ids){
        orgService.removeByIds(Arrays.asList(ids));
        return Result.succeed("ok");
    }

    @ApiOperation("获取机构信息")
    @PostMapping("/info/{id}")
    @RequiresRoles(value = {BizConstants.ROLE_ADMIN,BizConstants.ROLE_OPERATOR},logical = Logical.OR)
    @SysLog("获取机构信息")
    public Result<InstitutionModel> getInfoById(@PathVariable Long id){
        OrganizationPO po = orgService.getById(id);
        if (po == null){
            throw new BusinessException("请求错误，数据不存在");
        }
        InstitutionModel model = new InstitutionModel();
        BeanUtils.copyProperties(po,model);
        return Result.succeed(model);
    }

    @ApiOperation("查询机构信息")
    @PostMapping("/list")
    @RequiresRoles(value = {BizConstants.ROLE_ADMIN,BizConstants.ROLE_OPERATOR},logical = Logical.OR)
    @SysLog("查询机构信息")
    public PageResult<InstitutionModel> list(@RequestBody QueryOrgModel param){
        param.setType(OrgTypeEnum.INSTITUTION);
        PageResult<OrganizationPO> pageList = orgService.queryPageList(param);
        List<InstitutionModel> list = Lists.transform(pageList.getData(),(org) -> {
            InstitutionModel model = new InstitutionModel();
            BeanUtils.copyProperties(org,model);
            return model;
        });
        return PageResult.<InstitutionModel>builder().code(CodeEnum.SUCCESS.getCode()).count(pageList.getCount()).data(list).build();
    }
}
