package com.qkdata.common.jwt;

import com.auth0.jwt.JWT;
import com.auth0.jwt.JWTCreator;
import com.auth0.jwt.JWTVerifier;
import com.auth0.jwt.algorithms.Algorithm;
import com.auth0.jwt.interfaces.DecodedJWT;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

@Component
@EnableConfigurationProperties(JWTProperties.class)
public class JWTService {

    @Autowired
    private JWTProperties jwtProperties;

    /**
     * 头部信息
     *
     * @return
     */
    private void buildHeader(JWTCreator.Builder builder) {
        Map<String, Object> header = new HashMap<>();
        header.put("typ", jwtProperties.getTyp());
        header.put("alg", jwtProperties.getAlg());
        builder.withHeader(header);
    }

    /**
     * 载荷信息
     */
    private void buildPayload(JWTCreator.Builder builder, Map<String, Object> claims) {
        LocalDateTime now = LocalDateTime.now();
        builder.withIssuer(jwtProperties.getIss())
               .withAudience(jwtProperties.getAud())
               .withIssuedAt(issuedAt((now)))
               .withExpiresAt(expiresAt(now, jwtProperties.getExp()));
        if (claims != null) {
            for (Map.Entry<String, Object> entry : claims.entrySet()) {
                Object value = entry.getValue();
                if (value instanceof Boolean) {
                    builder.withClaim(entry.getKey(), (Boolean) value);
                } else if (value instanceof Integer) {
                    builder.withClaim(entry.getKey(), (Integer) value);
                } else if (value instanceof Long) {
                    builder.withClaim(entry.getKey(), (Long) value);
                } else if (value instanceof String) {
                    builder.withClaim(entry.getKey(), (String) value);
                } else if (value instanceof Double) {
                    builder.withClaim(entry.getKey(), (Double) value);
                } else if (value instanceof Date) {
                    builder.withClaim(entry.getKey(), (Date) value);
                } else {
                    throw new IllegalStateException("jwt payload claims type mismatch");
                }
            }
        }

    }

    /**
     * 签名
     *
     * @param builder
     * @return jwt字符串
     */
    private String buildSignature(JWTCreator.Builder builder) {
        return builder.sign(Algorithm.HMAC256(jwtProperties.getSecret()));
    }

    /**
     * 创建JWT
     *
     * @param claims 业务参数
     * @return
     */
    public String createJWT(Map<String, Object> claims) {
        JWTCreator.Builder builder = JWT.create();
        buildHeader(builder);
        buildPayload(builder, claims);
        return buildSignature(builder);
    }

    public String createJWT() {
        return createJWT(null);
    }


    /**
     * 签发时间
     *
     * @param issuedAt
     * @return
     */
    private Date issuedAt(LocalDateTime issuedAt) {
        return Date.from(issuedAt.atZone(ZoneId.systemDefault()).toInstant());
    }

    /**
     * 过期时间
     *
     * @param issuedAt
     * @param hours
     * @return
     */
    private Date expiresAt(LocalDateTime issuedAt, Integer hours) {
        return Date.from(issuedAt.plusHours(hours).atZone(ZoneId.systemDefault()).toInstant());
    }

    /**
     * 解码
     *
     * @param jwt
     */
    public DecodedJWT decode(String jwt) {
        JWTVerifier verifier = JWT.require(Algorithm.HMAC256(jwtProperties.getSecret())).build();
        return verifier.verify(jwt);
    }

}
