package com.qkdata.common.config;

import com.qkdata.common.base.exception.BusinessException;
import com.qkdata.common.base.model.Result;
import com.qkdata.common.oauth.AuthorizationResponseEnum;
import lombok.extern.slf4j.Slf4j;
import org.apache.shiro.authz.UnauthenticatedException;
import org.apache.shiro.authz.UnauthorizedException;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.validation.BindingResult;
import org.springframework.web.HttpMediaTypeNotSupportedException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.MissingPathVariableException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.method.annotation.MethodArgumentTypeMismatchException;
import org.springframework.web.servlet.NoHandlerFoundException;

import javax.validation.ConstraintViolationException;

@RestControllerAdvice
@Slf4j
public class GlobalExceptionHandle {

    @ExceptionHandler(value = MethodArgumentNotValidException.class)
    public ResponseEntity<Result> bindExceptionHandler(MethodArgumentNotValidException e) {
        BindingResult bindingResult = e.getBindingResult();
        String message = bindingResult.getFieldError().getDefaultMessage();
        log.warn("方法参数无效: {}", message);
        return ResponseEntity.status(HttpStatus.OK).body(Result.failed(message));
    }

    @ExceptionHandler(value = HttpRequestMethodNotSupportedException.class)
    public ResponseEntity<Result> httpRequestMethodNotSupportedExceptionHandler(HttpRequestMethodNotSupportedException e) {
        log.warn("不支持的Method: {}", e.getMessage());
        return ResponseEntity.status(HttpStatus.NOT_FOUND).body(Result.failed("不支持当前请求方法"));
    }

    @ExceptionHandler(value = Exception.class)
    public ResponseEntity<Result> handler(Exception e) {
        log.error("服务器错误", e);
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(Result.failed("服务器处理错误"));
    }


    @ExceptionHandler(value = BusinessException.class)
    public ResponseEntity<Result> businessExceptionhandler(BusinessException e) {
        log.warn("业务异常, message: {}", e.getMessage());
        return ResponseEntity.status(HttpStatus.OK).body(Result.failed(e.getMessage()));
    }

    @ExceptionHandler(value = HttpMessageNotReadableException.class)
    public ResponseEntity<Result> httpMessageConvertExceptionHanlder(HttpMessageNotReadableException e) {
        log.warn("无法解析请求参数: {}", e.getMessage());
        return ResponseEntity.badRequest().body(Result.failed(e.getMessage()));
    }

    @ExceptionHandler(value = MissingPathVariableException.class)
    public ResponseEntity<Result> missingPathVariableExceptionHanlder(MissingPathVariableException e) {
        log.warn("请求链接缺少参数: {}", e.getMessage());
        return ResponseEntity.badRequest().body(Result.failed(e.getMessage()));
    }

    @ExceptionHandler(value = MissingServletRequestParameterException.class)
    public ResponseEntity<Result> requestParameterExceptionHandler(MissingServletRequestParameterException e) {
        log.warn("缺少必需的请求参数: {}", e.getMessage());
        return ResponseEntity.badRequest().body(Result.failed(e.getMessage()));
    }

    @ExceptionHandler(value = MethodArgumentTypeMismatchException.class)
    public ResponseEntity<Result> methodArgumentTypeMismatchExceptionHandler(MethodArgumentTypeMismatchException e) {
        log.warn("参数类别转换失败: {}", e.getMessage());
        return ResponseEntity.badRequest().body(Result.failed(e.getMessage()));
    }

    @ExceptionHandler(value = ConstraintViolationException.class)
    public ResponseEntity<Result> constraintViolationExceptionHandler(ConstraintViolationException e) {
        log.warn("参数校验异常: {}", e.getMessage());
        return ResponseEntity.badRequest().body(Result.failed(e.getMessage()));
    }

    @ExceptionHandler(value = HttpMediaTypeNotSupportedException.class)
    public ResponseEntity<Result> httpMediaTypeNotSupportedExceptionHandler(HttpMediaTypeNotSupportedException e) {
        log.warn("不支持MediaType: {}", e.getMessage());
        return ResponseEntity.badRequest().body(Result.failed(e.getMessage()));
    }

    @ExceptionHandler(value = NoHandlerFoundException.class)
    public ResponseEntity<Result> noHandlerFoundExceptionHandler(NoHandlerFoundException e) {
        log.warn("no handler found: {}", e.getMessage());
        return ResponseEntity.status(HttpStatus.NOT_FOUND).body(Result.failed(e.getMessage()));
    }
    @ExceptionHandler(value = UnauthorizedException.class)
    public ResponseEntity<Result> unauthorizedExceptionHandler(UnauthorizedException e) {
        log.warn("no permission found: {}", e.getMessage());
        return ResponseEntity.status(HttpStatus.FORBIDDEN).body(Result.failed("无权限"));
    }
    @ExceptionHandler(value = UnauthenticatedException.class)
    public ResponseEntity<Result> unauthenticatedExceptionHandler(UnauthenticatedException e){
        log.warn("no permission found: {}", e.getMessage());
        return ResponseEntity.status(HttpStatus.OK).body(Result.of("", AuthorizationResponseEnum.INVALID_TOKEN.code(),AuthorizationResponseEnum.INVALID_TOKEN.text()));
    }


}
