package com.qkdata.biz.management.service;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.RandomUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSONObject;
import com.alipay.api.AlipayApiException;
import com.alipay.api.domain.AlipayTradeWapPayModel;
import com.alipay.api.response.AlipayTradeCloseResponse;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.github.binarywang.wxpay.bean.request.WxPayUnifiedOrderRequest;
import com.github.binarywang.wxpay.bean.result.WxPayOrderCloseResult;
import com.github.binarywang.wxpay.bean.result.WxPayOrderQueryResult;
import com.github.binarywang.wxpay.exception.WxPayException;
import com.github.binarywang.wxpay.service.WxPayService;
import com.qkdata.alipay.service.AlipayService;
import com.qkdata.biz.common.BizConstants;
import com.qkdata.biz.enums.*;
import com.qkdata.biz.management.entity.CoursePO;
import com.qkdata.biz.management.entity.ProductOrderPO;
import com.qkdata.biz.management.mapper.ProductOrderMapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.qkdata.biz.management.vo.ProductOrderModel;
import com.qkdata.biz.management.vo.QueryProductOrderModel;
import com.qkdata.biz.sys.service.SysConfigService;
import com.qkdata.biz.sys.service.SysRoleService;
import com.qkdata.biz.sys.service.SysUserService;
import com.qkdata.biz.sys.vo.SysRoleModel;
import com.qkdata.biz.web.vo.*;
import com.qkdata.common.base.entity.BasePO;
import com.qkdata.common.base.enums.CodeEnum;
import com.qkdata.common.base.exception.BusinessException;
import com.qkdata.common.base.model.PageResult;
import com.qkdata.common.base.model.Result;
import com.qkdata.common.util.HttpContextUtils;
import com.qkdata.common.util.IPUtils;
import com.qkdata.common.util.UserContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.List;

/**
 * <p>
 * 产品订单 服务类
 * </p>
 *
 * @author liuyang
 * @since 2021-06-02
 */
@Service
public class ProductOrderService extends ServiceImpl<ProductOrderMapper, ProductOrderPO> {

    @Autowired
    private WxPayService wxService;
    @Autowired
    private CourseService courseService;
    @Autowired
    private SysConfigService sysConfigService;
    @Autowired
    private SysUserService sysUserService;
    @Value("${frontend.domain}")
    private String frontendDomain;
    @Autowired
    private SysRoleService roleService;
    @Autowired
    private UserCourseAuthService userCourseAuthService;
    @Autowired
    private OrgSurplusService orgSurplusService;
    @Autowired
    private AlipayService alipayService;
    @Autowired
    @Qualifier("stringRedisTemplate")
    private RedisTemplate redisTemplate;

    private String generateOrderNo() {
        DateTime nowDateTime = DateUtil.date();
        String dateStr = DateUtil.format(nowDateTime,"yyyyMMddHHmmssSSS");
        return dateStr + RandomUtil.randomNumbers(4);
    }

    private <T> T wxCreateOrder(ProductOrderPO order, PaySourceEnum source,String openId) throws WxPayException {
        WxPayUnifiedOrderRequest payRequest = new WxPayUnifiedOrderRequest();
        if(wxService.getConfig().isUseSandboxEnv()){
            String key = wxService.getSandboxSignKey();
            wxService.getConfig().setMchKey(key);
        }
        payRequest.setNonceStr(RandomUtil.randomString(32));
        payRequest.setBody(StrUtil.format("在线培训网站-{}",order.getOrderName()));
        payRequest.setOutTradeNo(order.getOrderNo());
        //金额元转为分
        BigDecimal fee = order.getPaymentMoney().multiply(new BigDecimal(100));
        if (wxService.getConfig().isUseSandboxEnv()){
            payRequest.setTotalFee(101);//成功用例
//            payRequest.setTotalFee(102);//失败用例
        }else {
            payRequest.setTotalFee(fee.intValue());
        }
        payRequest.setSpbillCreateIp(IPUtils.getIpAddr(HttpContextUtils.getHttpServletRequest()));
        payRequest.setNotifyUrl(StrUtil.format("{}{}",frontendDomain,"/online-edu-backend/wx/pay/notify/order"));
        payRequest.setTradeType(source.name());
        if (source == PaySourceEnum.JSAPI){
            payRequest.setOpenid(openId);
        }
        return wxService.createOrder(payRequest);
    }

    @Transactional
    public CreateOrderResult createOrder(CreateOrderModel model) {
        CreateOrderResult result = new CreateOrderResult();
        ProductOrderPO orderPO = createPO(model);
        save(orderPO);
        result.setOrderNo(orderPO.getOrderNo());
        Object rtObj = prePay(orderPO,model.getSource(),model.getOpenId(),model.getReturnUrl());
        result.setResult(rtObj);
        return result;
    }
    private Object prePay(ProductOrderPO orderPO,PaySourceEnum source,String openId,String returnUrl){
        if (orderPO.getPayMethod() == PayMethodEnum.WeiXin){
            try {
                Object rtObj = wxCreateOrder(orderPO,source,openId);
                return rtObj;
            } catch (WxPayException e) {
                log.error(e.getMessage(),e);
                throw new BusinessException("微信支付异常");
            }
        }else if (orderPO.getPayMethod() == PayMethodEnum.AliPay){
            AlipayTradeWapPayModel payModel = new AlipayTradeWapPayModel();
            payModel.setOutTradeNo(orderPO.getOrderNo());
            payModel.setSubject(orderPO.getOrderName());
            payModel.setTotalAmount(orderPO.getPaymentMoney().toString());
            payModel.setQuitUrl(returnUrl+"&result=true");
            payModel.setProductCode("QUICK_WAP_WAY");
            try {
                String form = alipayService.prePayOrder(payModel);
                return form;
            } catch (AlipayApiException e) {
                log.error(e.getMessage(),e);
                throw new BusinessException("支付宝创建订单异常");
            }
        }
        return null;
    }

    private ProductOrderPO createPO(CreateOrderModel model){
        ProductOrderPO po = new ProductOrderPO();
        po.setOrderNo(generateOrderNo());
        po.setProductType(model.getProductType());
        FullUserInfo userInfo = sysUserService.findFullUserInfo(model.getUsername());
        switch (po.getProductType()){
            case USER_VIP:
                po.setOrderName("会员包月");
                String vipPriceStr = sysConfigService.getValueByKey(BizConstants.CONFIG_KEY_VIP_PRICE);
                if (StrUtil.isBlank(vipPriceStr)){
                    throw new BusinessException("处理错误，尚未配置会员价格");
                }
                po.setProductPrice(new BigDecimal(vipPriceStr));
                break;
            case ENTERPRISE_VIP:
                po.setOrderName("企业会员");
                String enterpriseVipPriceStr = sysConfigService.getValueByKey(BizConstants.CONFIG_KEY_ENTERPRICE_VIP_PRICE);
                if (StrUtil.isBlank(enterpriseVipPriceStr)){
                    throw new BusinessException("处理错误，尚未配置企业会员价格");
                }
                po.setProductPrice(new BigDecimal(enterpriseVipPriceStr));
                po.setOrgId(userInfo.getEnterpriseId());
                break;
            case ENTERPRISE_COURSE_BUY:case COURSE_BUY:
                CoursePO coursePO = courseService.getById(model.getCourseId());
                if (coursePO == null){
                    throw new BusinessException("请求错误，课程不存在");
                }
                String orderName = "付费点播：{}";
                po.setOrderName(StrUtil.format(orderName,coursePO.getName()));
                if (userInfo.getType() == AccountTypeEnum.USER){
                    po.setProductPrice(coursePO.getPrice());
                }else {
                    po.setProductPrice(coursePO.getVipPrice());
                }
                po.setOrgId(userInfo.getEnterpriseId());
                break;
            default:
                throw new BusinessException("购买产品类型错误");
        }
        po.setProductCount(model.getCount());
        po.setOrderMoney(po.getProductPrice().multiply(new BigDecimal(po.getProductCount())));
        po.setPaymentMoney(po.getOrderMoney());
        po.setCourseId(model.getCourseId());
        po.setPayMethod(model.getPayMethod());
        po.setStatus(ProductOrderStatusEnum.WAIT_PAY);
        po.setUserId(UserContext.getUserId());
        po.setSource(model.getSource());
        return po;
    }


    public void updateStatus(String outTradeNo, ProductOrderStatusEnum status) {
        ProductOrderPO po = new ProductOrderPO();
        po.setStatus(status);
        update(po, Wrappers.<ProductOrderPO>lambdaUpdate().eq(ProductOrderPO::getOrderNo,outTradeNo));
    }

    public ProductOrderStatusEnum queryFromThirdPlatform(String orderNo,PayMethodEnum payMethod){
        try {
            if (payMethod == PayMethodEnum.WeiXin) {
                if(wxService.getConfig().isUseSandboxEnv()){
                    String key = wxService.getSandboxSignKey();
                    wxService.getConfig().setMchKey(key);
                }
                WxPayOrderQueryResult queryResult = wxService.queryOrder(null, orderNo);
                if ("SUCCESS".equals(queryResult.getTradeState())){
                    updateStatus(orderNo,ProductOrderStatusEnum.PAY_COMPLETE);
                    return ProductOrderStatusEnum.PAY_COMPLETE;
                }else if ("CLOSED".equals(queryResult.getTradeState())){
                    updateStatus(orderNo,ProductOrderStatusEnum.PAY_CLOSE);
                    return ProductOrderStatusEnum.PAY_CLOSE;
                }else {
                    return ProductOrderStatusEnum.WAIT_PAY;
                }
            }else if (payMethod == PayMethodEnum.AliPay){
                try {
                    String queryResult = alipayService.queryOrder(orderNo);
                    if (queryResult.equals("TRADE_SUCCESS") || queryResult.equals("TRADE_FINISHED")){
                        updateStatus(orderNo,ProductOrderStatusEnum.PAY_COMPLETE);
                        return ProductOrderStatusEnum.PAY_COMPLETE;
                    }else if (queryResult.equals("TRADE_CLOSED")){
                        updateStatus(orderNo,ProductOrderStatusEnum.PAY_CLOSE);
                        return ProductOrderStatusEnum.PAY_CLOSE;
                    }else {
                        return ProductOrderStatusEnum.WAIT_PAY;
                    }
                } catch (AlipayApiException e) {
                    throw new BusinessException("查询支付宝订单请求异常");
                }
            }else {
                throw new BusinessException("暂不支持的支付类型");
            }
        } catch (WxPayException e) {
            log.error(e.getMessage(),e);
            return ProductOrderStatusEnum.WAIT_PAY;
        }
    }

    public ProductOrderStatusEnum queryStatus(String orderNo) {
        ProductOrderPO orderPO = getByOrderNo(orderNo);
        if (orderPO == null){
            throw new BusinessException("订单号不存在");
        }
        if (orderPO.getStatus() == ProductOrderStatusEnum.WAIT_PAY){
            return queryFromThirdPlatform(orderNo,orderPO.getPayMethod());
        }else {
            return orderPO.getStatus();
        }

    }

    private ProductOrderPO getByOrderNo(String orderNo) {
        return getOne(Wrappers.<ProductOrderPO>lambdaQuery().eq(ProductOrderPO::getOrderNo,orderNo));
    }

    public PageResult<ProductOrderModel> queryPageList(QueryProductOrderModel model) {
        Page page = new Page(model.getPageIndex(),model.getPageSize());
        List<ProductOrderModel> list = baseMapper.queryPageList(page,model);
        for (ProductOrderModel orderModel : list){
            List<SysRoleModel> roleList = roleService.getUserRoles(orderModel.getUserId());
            if (CollUtil.isNotEmpty(roleList)){
                orderModel.setUserRoleName(roleList.get(0).getName());
            }
        }
        return PageResult.<ProductOrderModel>builder().code(CodeEnum.SUCCESS.getCode()).count(page.getTotal()).data(list).build();
    }

    public void orderComplete(String orderNo) {
        updateStatus(orderNo,ProductOrderStatusEnum.PAY_COMPLETE);
        ProductOrderPO orderPO = getByOrderNo(orderNo);
        if (orderPO.getProductType() == ProductTypeEnum.USER_VIP){
            LocalDateTime payCompleteTime = orderPO.getUpdateTime();//支付完成时间,即生效时间
            int month = orderPO.getProductCount();//购买的几个月VIP
            Long userId = orderPO.getUserId();
            sysUserService.updateVIPTimeLimit(userId,payCompleteTime,month);

        }else if (orderPO.getProductType() == ProductTypeEnum.ENTERPRISE_VIP){
            Long orgId = sysUserService.getUserEnterpriesId(orderPO.getUserId());
            if (orgId != null){
                orgSurplusService.updateSurplus(orgId,null,orderPO.getProductType(),orderPO.getProductCount());
            }
        }else if (orderPO.getProductType() == ProductTypeEnum.COURSE_BUY){
            userCourseAuthService.updateAuth(orderPO.getUserId(),orderPO.getCourseId(),orderPO.getUpdateTime());
        }else if (orderPO.getProductType() == ProductTypeEnum.ENTERPRISE_COURSE_BUY){
            Long orgId = sysUserService.getUserEnterpriesId(orderPO.getUserId());
            if (orgId != null){
                orgSurplusService.updateSurplus(orgId,orderPO.getCourseId(),orderPO.getProductType(),orderPO.getProductCount());
            }

        }

    }


    public CreateOrderResult payAgain(CreateOrderModel model) {
        CreateOrderResult result = new CreateOrderResult();
        result.setOrderNo(model.getOrderNo());
        ProductOrderPO orderPO = getByOrderNo(model.getOrderNo());
        if (orderPO == null){
            throw new BusinessException("请求错误，订单不存在");
        }
        if (orderPO.getStatus() == ProductOrderStatusEnum.PAY_COMPLETE){
            throw new BusinessException("订单已完成，请不要重覆支付");
        }
        //下单来源不一致
        if (orderPO.getPayMethod() == PayMethodEnum.WeiXin && orderPO.getSource() != model.getSource()){
            if (orderPO.getSource() == PaySourceEnum.MWEB){
                throw new BusinessException("请从浏览器中支付");
            }else if (orderPO.getSource() == PaySourceEnum.JSAPI){
                throw new BusinessException("请从微信浏览器中支付");
            }
        }
        ProductOrderStatusEnum status = queryFromThirdPlatform(orderPO.getOrderNo(),orderPO.getPayMethod());
        if (status == ProductOrderStatusEnum.WAIT_PAY){
            Object rtObj = prePay(orderPO,model.getSource(),model.getOpenId(),model.getReturnUrl());
            result.setResult(rtObj);
            return result;
        }else if (status == ProductOrderStatusEnum.PAY_COMPLETE){
            throw new BusinessException("订单已支付完成，请不要重覆下单");
        }else if (status == ProductOrderStatusEnum.PAY_CLOSE){
            throw new BusinessException("订单已关闭，请重新下单");
        }
        return result;
    }

    /**
     * 定时检查未支付订单的状态(取30分钟前的订单)
     */
    public void checkProductOrder() {
        LocalDateTime now = LocalDateTime.now();
        LocalDateTime endTime = now.minusMinutes(30);
        List<ProductOrderPO> orderList = list(Wrappers.<ProductOrderPO>lambdaQuery()
                .eq(ProductOrderPO::getStatus,ProductOrderStatusEnum.WAIT_PAY)
                .le(BasePO::getCreateTime,endTime));
        for (ProductOrderPO order : orderList){
            ProductOrderStatusEnum status = queryFromThirdPlatform(order.getOrderNo(),order.getPayMethod());
            if (status == ProductOrderStatusEnum.WAIT_PAY){
                closeOrder(order.getOrderNo(),order.getPayMethod());
            }
        }
    }

    private void closeOrder(String orderNo,PayMethodEnum payMethod) {
        if (payMethod == PayMethodEnum.WeiXin){
            try {
                WxPayOrderCloseResult result = wxService.closeOrder(orderNo);
                if (result.getResultCode().equals("SUCCESS")){
                    if (result.getResultCode().equals("SUCCESS")){
                        //关单成功
                        updateStatus(orderNo,ProductOrderStatusEnum.PAY_CLOSE);
                    }else if (result.getErrCode().equals("ORDERPAID")){
                        //订单已支付
                        updateStatus(orderNo,ProductOrderStatusEnum.PAY_COMPLETE);
                    }else if (result.getErrCode().equals("ORDERCLOSED")){
                        //订单已关闭
                        updateStatus(orderNo,ProductOrderStatusEnum.PAY_CLOSE);
                    }
                }else {
                    log.error(JSONObject.toJSONString(result));
                }
            } catch (WxPayException e) {
                log.error(e.getMessage(),e);
            }
        }else if (payMethod == PayMethodEnum.AliPay){
            try {
                AlipayTradeCloseResponse response = alipayService.closeOrder(orderNo);
                if (response.isSuccess()){
                    //关单成功
                    updateStatus(orderNo,ProductOrderStatusEnum.PAY_CLOSE);
                }else {
                    //关单失败
                    log.error(JSONObject.toJSONString(response));
                }
            } catch (AlipayApiException e) {
                log.error(e.getMessage(),e);
            }

        }

    }
}