package com.qkdata.aliyun.service;

import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSONObject;
import com.aliyun.oss.OSS;
import com.aliyun.oss.OSSClientBuilder;
import com.aliyun.oss.common.utils.BinaryUtil;
import com.aliyun.oss.model.MatchMode;
import com.aliyun.oss.model.PolicyConditions;
import com.aliyuncs.DefaultAcsClient;
import com.aliyuncs.exceptions.ClientException;
import com.aliyuncs.profile.DefaultProfile;
import com.aliyuncs.vod.model.v20170321.*;
import com.google.common.collect.Maps;
import com.qkdata.aliyun.config.AliyunConfig;
import com.qkdata.aliyun.vo.AliyunOSSUploadPolicyModel;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.util.Date;
import java.util.Map;

/**
 * 阿里云接口封装
 */
@Service
public class AliyunService {
    @Autowired
    private AliyunConfig aliyunConfig;
    @Value("${frontend.domain}")
    private String frontendDomain;
    @Value("${server.servlet.context-path}")
    private String serverContextPath;
    private OSS ossClient;
    private DefaultAcsClient vodClient;
    /**
     * 过期时间（秒）
     */
    private long expireTime = 60;

    /**
     * 初始化
     */
    @PostConstruct
    private void init(){
        if (ossClient == null){
            ossClient = new OSSClientBuilder().build(aliyunConfig.getEndpoint(),aliyunConfig.getAccessKeyId(),aliyunConfig.getAccessKeySecret());
        }
        if (vodClient == null){
            vodClient = initVodClient();
        }
    }

    /**
     * 消毁
     */
    @PreDestroy
    private void clean(){
        if (ossClient != null){
            ossClient.shutdown();
        }
        if (vodClient != null){
            vodClient.shutdown();
        }
    }

    /**
     * 初始化vod client
     * @return
     */
    private DefaultAcsClient initVodClient() {
        DefaultProfile profile = DefaultProfile.getProfile(aliyunConfig.getRegion(),aliyunConfig.getAccessKeyId(),aliyunConfig.getAccessKeySecret());
        DefaultAcsClient client = new DefaultAcsClient(profile);
        return client;
    }

    /**
     * 获取上传策略参数
     * @return
     * @throws UnsupportedEncodingException
     */
    public AliyunOSSUploadPolicyModel getUploadPolicy() throws UnsupportedEncodingException {
        long expireEndTime = System.currentTimeMillis() + expireTime * 1000;
        Date expiration = new Date(expireEndTime);
        PolicyConditions policyConditions = new PolicyConditions();
        policyConditions.addConditionItem(PolicyConditions.COND_CONTENT_LENGTH_RANGE, 0, 1048576000);
        policyConditions.addConditionItem(MatchMode.StartWith, PolicyConditions.COND_KEY, "");

        String postPolicy = ossClient.generatePostPolicy(expiration,policyConditions);
        byte[] binaryData = postPolicy.getBytes("utf-8");
        String encodedPolicy = BinaryUtil.toBase64String(binaryData);
        String postSignature = ossClient.calculatePostSignature(postPolicy);

        AliyunOSSUploadPolicyModel model = new AliyunOSSUploadPolicyModel();
        model.setAccessId(aliyunConfig.getAccessKeyId());
        model.setPolicy(encodedPolicy);
        model.setSignature(postSignature);
        model.setHost("https://" + aliyunConfig.getBucket() + "." + aliyunConfig.getEndpoint());
        model.setExpire(String.valueOf(expireEndTime / 1000));
        model.setDir("");

        JSONObject jsonCallback = new JSONObject();
        jsonCallback.put("callbackUrl",frontendDomain + serverContextPath + "/aliyun/oss/callback");
        jsonCallback.put("callbackBody","{\"bucket\":${bucket},\"imageWidth\":\"${imageInfo.width}\",\"imageFormat\":${imageInfo.format},\"fileName\":${object},\"size\":\"${size}\",\"etag\":${etag},\"mimeType\":${mimeType},\"imageHeight\":\"${imageInfo.height}\"}");
        jsonCallback.put("callbackBodyType","application/json");
        String base64CallbackBody = BinaryUtil.toBase64String(jsonCallback.toString().getBytes());
        model.setCallback(base64CallbackBody);
        return model;
    }

    /**
     * 生成阿里云oss访问链接
     * @param fileName
     * @return
     */
    public String generateUrl(String fileName) {
        // 设置URL过期时间为1小时。
        Date expiration = new Date(new Date().getTime() + expireTime * 1000);
        // 生成以GET方法访问的签名URL，访客可以直接通过浏览器访问相关内容。
        URL url = ossClient.generatePresignedUrl(aliyunConfig.getBucket(), fileName, expiration);
        return url.toString();
    }

    /**
     * 获取视频上传地址和凭证
     * @return CreateUploadVideoResponse 获取视频上传地址和凭证响应数据
     * @throws Exception
     */
    public CreateUploadVideoResponse createUploadVideo(String title,String fileName) throws Exception {
        CreateUploadVideoRequest request = new CreateUploadVideoRequest();
        request.setTitle(title);
        request.setFileName(fileName);

        //UserData，用户自定义设置参数，用户需要单独回调URL及数据透传时设置(非必须)
        JSONObject userData = new JSONObject();

        //UserData回调部分设置
        JSONObject messageCallback = new JSONObject();
        messageCallback.put("CallbackURL", frontendDomain + serverContextPath + "/aliyun/video/callback");
        messageCallback.put("CallbackType", "http");
        userData.put("MessageCallback", messageCallback.toJSONString());

        //UserData透传数据部分设置
        JSONObject extend = new JSONObject();
        extend.put("fileName", fileName);
        userData.put("Extend", extend.toJSONString());

        request.setUserData(userData.toJSONString());

        return vodClient.getAcsResponse(request);
    }
    /**
     * 刷新视频上传凭证
     * @param videoId
     * @return RefreshUploadVideoResponse 刷新视频上传凭证响应数据
     * @throws Exception
     */
    public RefreshUploadVideoResponse refreshUploadVideo(String videoId) throws Exception {
        RefreshUploadVideoRequest request = new RefreshUploadVideoRequest();
        request.setVideoId(videoId);
        return vodClient.getAcsResponse(request);
    }

    /**
     *
     * @return
     */
    public Map<String,String> getVideoConfig() {
        Map<String,String> map = Maps.newConcurrentMap();
        map.put("userId",aliyunConfig.getUserId());
        map.put("region",aliyunConfig.getRegion());
        return map;
    }

    /**
     * 上传文件
     * @param fullFileName
     * @param fileInputStream
     */
    public void uploadFile(String fullFileName, InputStream fileInputStream){
        ossClient.putObject(aliyunConfig.getBucket(),fullFileName,fileInputStream);
    }

    public String getPublicUrl() {
        if (StrUtil.isNotBlank(aliyunConfig.getCdn())){
            return aliyunConfig.getCdn();
        }else {
            return "https://"+aliyunConfig.getBucket()+"."+aliyunConfig.getEndpoint();
        }

    }

    /**
     * 获取视频信息
     * @param videoId
     * @return
     * @throws ClientException
     */
    public GetVideoInfoResponse getVideoInfo(String videoId) throws ClientException {
        GetVideoInfoRequest request = new GetVideoInfoRequest();
        request.setVideoId(videoId);
        return vodClient.getAcsResponse(request);
    }

    /**
     * 获取视频播放信息
     * @param videoId
     * @return
     * @throws ClientException
     */
    public GetPlayInfoResponse getVideoPlayInfo(String videoId) throws ClientException {
        GetPlayInfoRequest request = new GetPlayInfoRequest();
        request.setVideoId(videoId);
        return vodClient.getAcsResponse(request);

    }

    /**
     * 获取视频播认证信息
     * @param videoId
     * @return
     * @throws ClientException
     */
    public GetVideoPlayAuthResponse getVideoPlayAuth(String videoId) throws ClientException {
        GetVideoPlayAuthRequest request = new GetVideoPlayAuthRequest();
        request.setVideoId(videoId);
        request.setAuthInfoTimeout(3000L);
        return vodClient.getAcsResponse(request);
    }

    /**
     * 删除视频
     * @param videoId
     * @throws ClientException
     */
    public void removeVideo(String videoId) throws ClientException {
        DeleteMezzaninesRequest request = new DeleteMezzaninesRequest();
        request.setVideoIds(videoId);
        DeleteMezzaninesResponse response = vodClient.getAcsResponse(request);

    }
}
