package com.qkdata.biz.management.service;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.qkdata.biz.common.BizConstants;
import com.qkdata.biz.enums.AnswerTypeEnum;
import com.qkdata.biz.enums.QuestionStatusEnum;
import com.qkdata.biz.management.entity.AnswerPO;
import com.qkdata.biz.management.entity.QuestionPO;
import com.qkdata.biz.management.entity.TeacherPO;
import com.qkdata.biz.management.mapper.AnswerMapper;
import com.qkdata.biz.management.mapper.QuestionMapper;
import com.qkdata.biz.management.vo.*;
import com.qkdata.biz.sys.entity.SysUserPO;
import com.qkdata.biz.web.vo.AskModel;
import com.qkdata.common.base.enums.CodeEnum;
import com.qkdata.common.base.model.PageResult;
import com.qkdata.common.util.UserContext;
import lombok.extern.slf4j.Slf4j;
import org.apache.shiro.SecurityUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;

/**
 * 问答服务类
 */
@Slf4j
@Service
public class QAService {
    @Autowired
    private QuestionMapper questionMapper;
    @Autowired
    private AnswerMapper answerMapper;
    @Autowired
    private TeacherService teacherService;

    /**
     * 查询问题列表
     * @param param
     * @return
     */
    public PageResult<QuestionModel> questionPageList(QueryQuestionModel param) {
        Long teacherId = null;
        boolean isTeacher = SecurityUtils.getSubject().hasRole(BizConstants.ROLE_TEACHER);
        if (isTeacher){
            TeacherPO po = teacherService.getByAccount(UserContext.getUser().getUsername());
            teacherId = po.getId();
        }
        Page page = new Page(param.getPageIndex(),param.getPageSize());
        List<QuestionModel> list = questionMapper.questionPageList(page,param,teacherId);
        return PageResult.<QuestionModel>builder().code(CodeEnum.SUCCESS.getCode()).count(page.getTotal()).data(list).build();
    }

    /**
     * 获取问题回复列表
     * @param param
     * @return
     */
    public PageResult<QAModel> answerList(QueryAnswerModel param) {
        Page page = new Page(param.getPageIndex(),param.getPageSize());
        List<QAModel> list = answerMapper.answerPageList(page,param);
        return PageResult.<QAModel>builder().code(CodeEnum.SUCCESS.getCode()).count(page.getTotal()).data(list).build();
    }

    /**
     * 回复
     * @param model
     */
    @Transactional
    public void answer(AnswerModel model) {
        AnswerPO po = new AnswerPO();
        po.setQuestionId(model.getQuestionId());
        po.setContent(model.getContent());
        po.setUserId(UserContext.getUserId());
        boolean isTeacher = SecurityUtils.getSubject().hasRole(BizConstants.ROLE_TEACHER);
        if (isTeacher){
            po.setType(AnswerTypeEnum.TEACHER);
        }else {
            po.setType(AnswerTypeEnum.USER);
        }
        answerMapper.insert(po);
        //修改问题状态
        if (isTeacher){
            changeQuestionStatus(model.getQuestionId(), QuestionStatusEnum.REPLY);
        }else {
            changeQuestionStatus(model.getQuestionId(),QuestionStatusEnum.WAIT_REPLY);
        }

    }

    /**
     * 修改问题状态
     * @param questionId
     * @param status
     */
    public void changeQuestionStatus(Long questionId, QuestionStatusEnum status) {
        QuestionPO po = new QuestionPO();
        po.setId(questionId);
        po.setStatus(status);
        questionMapper.updateById(po);
    }

    /**
     * 用户提问
     * @param model
     */
    @Transactional
    public Long ask(AskModel model) {
        QuestionPO questionPO = new QuestionPO();
        questionPO.setUserId(UserContext.getUserId());
        questionPO.setCourseId(model.getCourseId());
        questionPO.setChapterId(model.getChapterId());
        questionPO.setContent(model.getContent());
        questionPO.setStatus(QuestionStatusEnum.WAIT_REPLY);
        questionMapper.insert(questionPO);
        AnswerPO answerPO = new AnswerPO();
        answerPO.setQuestionId(questionPO.getId());
        answerPO.setUserId(questionPO.getUserId());
        answerPO.setType(AnswerTypeEnum.USER);
        answerPO.setContent(questionPO.getContent());
        answerMapper.insert(answerPO);
        return questionPO.getId();
    }

    /**
     * 查询当前用户提问列表
     * @param param
     * @return
     */
    public PageResult<QuestionModel> myQuestionPageList(QueryQuestionModel param) {
        Page page = new Page(param.getPageIndex(),param.getPageSize());
        List<QuestionModel> list = questionMapper.myQuestionPageList(page,UserContext.getUserId());
        return PageResult.<QuestionModel>builder().code(CodeEnum.SUCCESS.getCode()).count(page.getTotal()).data(list).build();
    }

    /**
     * 获取问题信息
     * @param questionId
     * @return
     */
    public QuestionModel getQuestion(Long questionId) {
        return questionMapper.findQuestion(questionId);
    }
}
