package com.qkdata.biz.management.service;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.google.common.base.Function;
import com.google.common.collect.Lists;
import com.qkdata.biz.common.BizConstants;
import com.qkdata.biz.enums.AccountStatusEnum;
import com.qkdata.biz.enums.AccountTypeEnum;
import com.qkdata.biz.management.vo.MemberInfoModel;
import com.qkdata.biz.sys.entity.SysRolePO;
import com.qkdata.biz.sys.entity.SysUserExtPO;
import com.qkdata.biz.sys.entity.SysUserPO;
import com.qkdata.biz.sys.mapper.SysUserExtMapper;
import com.qkdata.biz.sys.service.SysRoleService;
import com.qkdata.biz.sys.service.SysUserRoleService;
import com.qkdata.biz.sys.service.SysUserService;
import com.qkdata.biz.management.vo.MemberListModel;
import com.qkdata.biz.management.vo.QueryMemberModel;
import com.qkdata.biz.sys.vo.SysRoleModel;
import com.qkdata.biz.sys.vo.SysUserModel;
import com.qkdata.biz.web.vo.FullUserInfo;
import com.qkdata.biz.web.vo.ModifyMemberInfoModel;
import com.qkdata.common.base.entity.BasePO;
import com.qkdata.common.base.enums.CodeEnum;
import com.qkdata.common.base.exception.BusinessException;
import com.qkdata.common.base.model.PageResult;
import com.qkdata.common.constants.Constants;
import com.qkdata.common.util.UserContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.stream.Collectors;

@Service
public class MemberService {
    @Autowired
    private SysUserService sysUserService;
    @Autowired
    private SysUserExtMapper sysUserExtMapper;
    @Autowired
    private SysRoleService sysRoleService;
    @Autowired
    private SysUserRoleService sysUserRoleService;
    @Autowired
    private UserTagService userTagService;

    public PageResult<MemberListModel> queryPage(QueryMemberModel param) {
        Page page = new Page(param.getPageIndex(),param.getPageSize());
        List<MemberListModel> list = sysUserExtMapper.queryPage(page,param);
        for (MemberListModel member : list){
            member.setUserTags(userTagService.findTagsByUserId(member.getId()));
        }
        return PageResult.<MemberListModel>builder().code(CodeEnum.SUCCESS.getCode()).count(page.getTotal()).data(list).build();
    }

    public boolean isEnterpriseAdminUser(String adminMobile) {
        boolean flag = false;
        SysUserPO sysUserPO = sysUserService.getByUsername(adminMobile);
        if (sysUserPO != null){
            //帐号已存在时，验证该帐号是否已为企业管理员
            List<SysRoleModel> roles = sysRoleService.getUserRoles(sysUserPO.getId());
            if (CollUtil.isNotEmpty(roles)) {
                for (SysRoleModel role : roles) {
                    if (role.getCode().equals(BizConstants.ROLE_ENTERPRISE_ADMIN)) {
                        flag = true;
                        break;
                    }
                }
            }
        }
        return flag;
    }

    public MemberInfoModel getByUsername(String username) {
        return getByUsername(username,false);
    }
    public MemberInfoModel getByUsername(String username, boolean withRole) {
        MemberInfoModel model = sysUserExtMapper.getByUsername(username);
        if (withRole){
            List<SysRoleModel> roleList = sysRoleService.getUserRoles(model.getId());
            List<String> roleCodeList = roleList.stream().map(SysRoleModel::getCode).collect(Collectors.toList());
            model.setRoleCodeList(roleCodeList);
        }
        return model;
    }

    @Transactional
    public void save(MemberInfoModel model) {
        FullUserInfo sysUser = sysUserService.findFullUserInfo(model.getUsername());
        if (sysUser == null){
            //创建帐号
            SysUserModel sysUserModel = convertToSysUserModel(model);
            Long userId = sysUserService.saveUser(sysUserModel);
            SysUserExtPO extUserPO = convertToExtUserPO(model);
            extUserPO.setUserId(userId);
            sysUserExtMapper.insert(extUserPO);
        }else {
            //更新用户信息
            if (sysUser.getEnterpriseId() != null){
                if (sysUser.getEnterpriseId().longValue() == model.getEnterpriseId().longValue()){
                    throw new BusinessException("该用户已是企业成员，不可重覆添加");
                }else {
                    throw new BusinessException("该用户已是其它企业成员，不可添加");
                }
            }
            //更新用户为企业会员
            SysUserExtPO userExtPO = sysUserExtMapper.selectOne(Wrappers.<SysUserExtPO>lambdaQuery().eq(SysUserExtPO::getUserId,sysUser.getId()));
            userExtPO.setName(model.getName());
            userExtPO.setEnterpriseId(model.getEnterpriseId());
            userExtPO.setType(AccountTypeEnum.ENTERPRISE_VIP);
            sysUserExtMapper.updateById(userExtPO);
        }

    }

    private SysUserExtPO convertToExtUserPO(MemberInfoModel model) {
        SysUserExtPO extUser = new SysUserExtPO();
        extUser.setName(model.getName());
        extUser.setSex(model.getSex());
        extUser.setAge(model.getAge() != null ? model.getAge() : null);
        extUser.setAvatarUrl(StrUtil.isNotBlank(model.getAvatarUrl()) ? model.getAvatarUrl() : null);
        extUser.setEnterpriseId(model.getEnterpriseId() != null ? model.getEnterpriseId() : null);
        extUser.setType(model.getType() != null ? model.getType() : null);
        return extUser;
    }

    private SysUserModel convertToSysUserModel(MemberInfoModel model) {
        SysUserModel sysUserModel = new SysUserModel();
        sysUserModel.setId(model.getId() != null ? model.getId() : null);
        sysUserModel.setUsername(model.getUsername());
        sysUserModel.setPassword(StrUtil.isNotBlank(model.getPassword()) ? model.getPassword() : Constants.PSAAWORD);
        sysUserModel.setNickName(model.getNickName());
        sysUserModel.setStatus(model.getStatus() == null ? AccountStatusEnum.UNACTIVATE : model.getStatus());
        List<SysRolePO> roleList = sysRoleService.getByCodes(model.getRoleCodeList());
        List<Long> roleIds = Lists.newArrayList();
        if (CollUtil.isNotEmpty(roleList)){
            roleIds = roleList.stream().map(SysRolePO::getId).collect(Collectors.toList());
        }
        sysUserModel.setRoleIdList(roleIds);
        return sysUserModel;
    }

    public void update(MemberInfoModel model) {
        SysUserModel sysUserModel = convertToSysUserModel(model);
        Long userId = sysUserService.updateUser(sysUserModel);
        SysUserExtPO extUserPO = convertToExtUserPO(model);
        extUserPO.setUserId(userId);
        sysUserExtMapper.update(extUserPO, Wrappers.<SysUserExtPO>lambdaUpdate().eq(SysUserExtPO::getUserId,userId));
    }

    public int countByEnterpriseId(Long enterpriseId) {
        int count = sysUserExtMapper.selectCount(Wrappers.<SysUserExtPO>lambdaQuery().eq(SysUserExtPO::getEnterpriseId,enterpriseId));
        return count;
    }

    /**
     * 删除企业成员
     * @param userId
     */
    @Transactional
    public void remove(Long userId) {
        sysUserExtMapper.update(null,Wrappers.<SysUserExtPO>lambdaUpdate()
                .set(SysUserExtPO::getType,AccountTypeEnum.USER)
                .set(SysUserExtPO::getEnterpriseId,null)
                .set(SysUserExtPO::getVipStartTime,null)
                .set(SysUserExtPO::getVipEndTime,null)
                .eq(SysUserExtPO::getUserId,userId));
        sysUserRoleService.saveOrUpdateUserRole(userId,Lists.newArrayList(5L));
    }

    public MemberInfoModel getByUserId(Long userId){
        return getByUserId(userId,false);
    }

    public MemberInfoModel getByUserId(Long userId, boolean withRole) {
        MemberInfoModel model = sysUserExtMapper.getByUserId(userId);
        if (withRole){
            List<SysRoleModel> roleList = sysRoleService.getUserRoles(model.getId());
            List<String> roleCodeList = roleList.stream().map(SysRoleModel::getCode).collect(Collectors.toList());
            model.setRoleCodeList(roleCodeList);
        }
        return model;
    }

    @Transactional
    public void updateInfo(ModifyMemberInfoModel model) {
        MemberInfoModel memberInfo = getByUserId(model.getId(),true);
        if (memberInfo == null){
            throw new BusinessException("请求错误，数据不存在");
        }
        if (!memberInfo.getUsername().equals(model.getUsername())){
            //修改手机号
            int count = sysUserService.count(Wrappers.<SysUserPO>lambdaQuery().eq(SysUserPO::getUsername,model.getUsername())
                    .ne(BasePO::getId, model.getId()));
            if (count > 0){
                throw new BusinessException("手机号已存在");
            }else {
                sysUserService.update(Wrappers.<SysUserPO>lambdaUpdate().set(SysUserPO::getUsername,model.getUsername()).eq(BasePO::getId,model.getId()));
            }
        }
        //修改姓名
        sysUserExtMapper.update(null,Wrappers.<SysUserExtPO>lambdaUpdate().set(SysUserExtPO::getName,model.getName()).eq(SysUserExtPO::getUserId,model.getId()));
        //修改角色
        List<String> roles = memberInfo.getRoleCodeList();
        boolean isManager = false;
        for (String roleCode : roles){
            if (roleCode.equals(BizConstants.ROLE_ENTERPRISE_ADMIN)){
                isManager = true;
                break;
            }
        }
        if (isManager != model.isManager()){
            List<Long> roleIds = Lists.newArrayList();
            if (model.isManager()){
                roleIds.add(3L);
            }else {
                roleIds.add(5L);
            }
            sysUserRoleService.saveOrUpdateUserRole(model.getId(),roleIds);
        }
    }

    public List<MemberInfoModel> findEnterpriseAdmin(Long enterpriseId) {
        return sysUserExtMapper.finfindEnterpriseAdmin(enterpriseId);
    }
}
