package com.qkdata.alipay.service;

import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSONObject;
import com.alipay.api.*;
import com.alipay.api.domain.AlipayTradeAppPayModel;
import com.alipay.api.domain.AlipayTradeCloseModel;
import com.alipay.api.domain.AlipayTradeQueryModel;
import com.alipay.api.domain.AlipayTradeWapPayModel;
import com.alipay.api.request.AlipayTradeAppPayRequest;
import com.alipay.api.request.AlipayTradeCloseRequest;
import com.alipay.api.request.AlipayTradeQueryRequest;
import com.alipay.api.request.AlipayTradeWapPayRequest;
import com.alipay.api.response.AlipayTradeAppPayResponse;
import com.alipay.api.response.AlipayTradeCloseResponse;
import com.alipay.api.response.AlipayTradeQueryResponse;
import com.alipay.api.response.AlipayTradeWapPayResponse;
import com.qkdata.alipay.config.AlipayProperties;
import com.qkdata.biz.enums.PaySourceEnum;
import com.qkdata.biz.management.entity.ProductOrderPO;
import com.qkdata.common.base.exception.BusinessException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;


/**
 * 支付宝接口封装
 */
@Slf4j
@Service
public class AlipayService {
    @Autowired
    private AlipayProperties alipayProperties;
    @Value("${frontend.domain}")
    private String frontendDomain;
    private AlipayClient alipayClient;

    /**
     * 初始化
     */
    @PostConstruct
    public void init(){
        alipayClient = new DefaultAlipayClient(alipayProperties.getUrl(),
                alipayProperties.getAppId(),
                alipayProperties.getAppPrivateKey(),
                AlipayConstants.FORMAT_JSON,
                AlipayConstants.CHARSET_UTF8,
                alipayProperties.getAlipayPublicKey(),
                AlipayConstants.SIGN_TYPE_RSA2);

    }

    /**
     * 统一下单调用
     * @param orderPO
     * @param  source
     * @param  returnUrl
     * @return
     * @throws AlipayApiException
     */
    public Object prePayOrder(ProductOrderPO orderPO, PaySourceEnum source, String returnUrl) throws AlipayApiException {
        AlipayRequest alipay_request=null;
        if (source == PaySourceEnum.MWEB){
            AlipayTradeWapPayModel requestObj =  new AlipayTradeWapPayModel();
            requestObj.setOutTradeNo(orderPO.getOrderNo());
            requestObj.setSubject(orderPO.getOrderName());
            requestObj.setTotalAmount(orderPO.getPaymentMoney().toString());
            requestObj.setProductCode("QUICK_WAP_WAY");
            requestObj.setQuitUrl(returnUrl+"&result=true");

            alipay_request = new AlipayTradeWapPayRequest();
            alipay_request.setBizModel(requestObj);
            alipay_request.setNotifyUrl(frontendDomain + "/online-edu-backend/alipay/notify/order");
            alipay_request.setReturnUrl(returnUrl+"&result=true");
        }else if (source == PaySourceEnum.APP){
            AlipayTradeAppPayModel requestObj =  new AlipayTradeAppPayModel();
            requestObj.setGoodsType("1");
            requestObj.setOutTradeNo(orderPO.getOrderNo());
            requestObj.setSubject(orderPO.getOrderName());
            requestObj.setTotalAmount(orderPO.getPaymentMoney().toString());
            requestObj.setProductCode("QUICK_MSECURITY_PAY");

            alipay_request = new AlipayTradeAppPayRequest();
            alipay_request.setBizModel(requestObj);
            alipay_request.setNotifyUrl(frontendDomain + "/online-edu-backend/alipay/notify/order");

        }else {
            throw new BusinessException("暂不支持该支付类型。source:"+source);
        }

        Object rtObj = null;
        AlipayResponse response = null;
        if (source == PaySourceEnum.MWEB){
            response = alipayClient.pageExecute(alipay_request);
        }else if (source == PaySourceEnum.APP){
            response = alipayClient.sdkExecute(alipay_request);
        }else {
            throw new BusinessException("暂不支持该支付类型。source:"+source);
        }
        if (response.isSuccess()){
            rtObj = response.getBody();
            return rtObj;
        }else {
            log.info("Alipay prePay request faild:"+ JSONObject.toJSONString(response));
        }
        return rtObj;
    }

    /**
     * 查询订单
     * @param orderNo
     * @return
     * @throws AlipayApiException
     */
    public String queryOrder(String orderNo) throws AlipayApiException {
        AlipayTradeQueryRequest request = new AlipayTradeQueryRequest();
        AlipayTradeQueryModel model = new AlipayTradeQueryModel();
        model.setOutTradeNo(orderNo);
        request.setBizModel(model);
        AlipayTradeQueryResponse response = alipayClient.execute(request);
        return response.getTradeStatus();
    }

    /**
     * 关闭订单
     * @param orderNo
     * @return
     * @throws AlipayApiException
     */
    public AlipayTradeCloseResponse closeOrder(String orderNo) throws AlipayApiException {
        AlipayTradeCloseRequest request = new AlipayTradeCloseRequest();
        AlipayTradeCloseModel model = new AlipayTradeCloseModel();
        model.setOutTradeNo(orderNo);
        request.setBizModel(model);
        AlipayTradeCloseResponse response = alipayClient.execute(request);
        return response;
    }
}
