package com.qkdata.biz.sys.service;

import cn.hutool.core.date.DateUtil;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Maps;
import com.qkdata.biz.enums.AccountStatusEnum;
import com.qkdata.biz.sys.entity.SysMenuPO;
import com.qkdata.biz.sys.entity.SysUserPO;
import com.qkdata.biz.sys.vo.LoginUserInfo;
import com.qkdata.biz.sys.vo.SysRoleModel;
import com.qkdata.common.base.exception.BusinessException;
import com.qkdata.common.jwt.JWTService;
import com.qkdata.common.oauth.AuthorizedUser;
import org.apache.shiro.crypto.hash.Sha256Hash;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

@Service
public class ShiroService {
    @Autowired
    private SysUserService sysUserService;
    @Autowired
    private SysMenuService sysMenuService;
    @Autowired
    private ObjectMapper objectMapper;
    @Autowired
    private JWTService jwtService;
    @Autowired
    private SysRoleService sysRoleService;

    public Set<String> getUserPermissions(Long userId) {
        List<String> permsList;

        //系统管理员，拥有最高权限
        if(userId == 1){
            List<SysMenuPO> menuList = sysMenuService.list();
            permsList = new ArrayList<>(menuList.size());
            for(SysMenuPO menu : menuList){
                permsList.add(menu.getPerms());
            }
        }else{
            permsList = sysUserService.queryAllPerms(userId);
        }
        //用户权限列表
        Set<String> permsSet = new HashSet<>();
        for(String perms : permsList){
            if(StringUtils.isEmpty(perms)){
                continue;
            }
            permsSet.addAll(Arrays.asList(perms.trim().split(",")));
        }
        return permsSet;
    }

    public LoginUserInfo login(String username, String password) throws JsonProcessingException {
        SysUserPO userPO = getUserByUserName(username);
        if(userPO == null || !userPO.getPassword().equals(new Sha256Hash(password, userPO.getSalt()).toHex())) {
            throw new BusinessException("用户名或密码错误");
        }
        if (userPO.getStatus() == AccountStatusEnum.DISABLE){
            throw new BusinessException("帐号已禁用");
        }
        if (userPO.getStatus() == AccountStatusEnum.UNACTIVATE){
            userPO.setActivateTime(DateUtil.date());
            sysUserService.updateById(userPO);
        }
        String token = generatorToken(userPO);
        LoginUserInfo loginUser = new LoginUserInfo();
        BeanUtils.copyProperties(userPO,loginUser);
        loginUser.setAuthorization(token);
        loginUser.setRoles(sysRoleService.getUserRoles(userPO.getId()));
        return loginUser;
    }

    private String generatorToken(SysUserPO userPO) throws JsonProcessingException {
        AuthorizedUser user = new AuthorizedUser();
        user.setUserId(userPO.getId());
        user.setUsername(userPO.getUsername());
        String userJson = objectMapper.writeValueAsString(user);
        Map<String,Object> userClaim = Maps.newConcurrentMap();
        userClaim.put("user",userJson);
        return jwtService.createJWT(userClaim);
    }

    public SysUserPO getUserByUserName(String username) {
        return sysUserService.getByUsername(username);
    }

    public Set<String> getUserRoles(Long userId) {
        List<SysRoleModel> roles = sysRoleService.getUserRoles(userId);
        return roles.stream().map(SysRoleModel::getCode).collect(Collectors.toSet());
    }
}
