package com.qkdata.biz.web.service;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.qkdata.biz.enums.AccountTypeEnum;
import com.qkdata.biz.enums.CourseAllowEnum;
import com.qkdata.biz.enums.CourseChargeModelEnum;
import com.qkdata.biz.enums.CourseStatusEnum;
import com.qkdata.biz.management.entity.CoursePO;
import com.qkdata.biz.management.entity.UserCourseAuthPO;
import com.qkdata.biz.management.entity.UserFavoritesPO;
import com.qkdata.biz.management.service.CourseService;
import com.qkdata.biz.management.service.UserCourseAuthService;
import com.qkdata.biz.management.service.UserFavoritesService;
import com.qkdata.biz.management.vo.QueryFavoriteCourseModel;
import com.qkdata.biz.sys.service.SysUserService;
import com.qkdata.biz.web.vo.FullUserInfo;
import com.qkdata.biz.web.vo.SearchCourseResultModel;
import com.qkdata.common.base.exception.BusinessException;
import com.qkdata.common.base.model.PageResult;
import com.qkdata.common.util.UserContext;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.List;

@Slf4j
@Service
public class UserCenterService {
    @Autowired
    private SysUserService sysUserService;
    @Autowired
    private CourseService courseService;
    @Autowired
    private UserFavoritesService userFavoritesService;
    @Autowired
    private UserCourseAuthService userCourseAuthService;

    /**
     * 验证当前登陆用户是否有播放该课程和章节的权限
     * 课程可见范围是否指定企业，当前用户是否符合
     * 课程收费模式为免费，允许观看
     * 课程收费模式为会员费免，当前用户是否已购买会员，并且在有效期中
     * 课程收费模式为付费点播，当前用户是否已购买该课程，并且在效期中
     * @param courseId
     * @param chaperId
     */
    public void checkCoursePlayPerm(Long courseId, Long chaperId) {
        String username = UserContext.getUser().getUsername();
        Long userId = UserContext.getUserId();
        FullUserInfo userInfo = sysUserService.findFullUserInfo(username);
        CoursePO course = courseService.getById(courseId);
        if (course.getStatus() == CourseStatusEnum.DOWN){
            throw new BusinessException("该课程已下架");
        }
        if (course.getAllow() == CourseAllowEnum.SPECIFY){
            List<Long> orgIds = courseService.getAllowEnterpriseList(courseId);

            boolean allow = false;
            for (Long orgId : orgIds){
                if (userInfo.getEnterpriseId() != null){
                    if (orgId.longValue() == userInfo.getEnterpriseId().longValue()){
                        allow = true;
                        break;
                    }
                }
            }
            if (!allow){
                throw new BusinessException("当前课程指已定企业观看，您不符合条件，请联系客服务人员");
            }
        }else {
            if (course.getChargeModel() == CourseChargeModelEnum.FREE){
                return;
            }else if (course.getChargeModel() == CourseChargeModelEnum.VIP_FREE){
                if (userInfo.getType() == AccountTypeEnum.USER){
                    throw new BusinessException("您还不是会员，请先成为会员再观看");
                }
                LocalDateTime now = LocalDateTime.now();
                if (!(now.isAfter(userInfo.getVipStartTime()) && now.isBefore(userInfo.getVipEndTime()))){
                    throw new BusinessException("您的会员已过期，请续费后再观看");
                }
            }else if (course.getChargeModel() == CourseChargeModelEnum.PAY){
                UserCourseAuthPO authPO = userCourseAuthService.getOne(Wrappers.<UserCourseAuthPO>lambdaQuery().eq(UserCourseAuthPO::getUserId,userId).eq(UserCourseAuthPO::getCourseId,courseId));
                if (authPO == null{
                    throw new BusinessException("当前课程为付费课程，请购买后再观看");
                }
                LocalDateTime now = LocalDateTime.now();
                if (!(now.isAfter(authPO.getStartTime()) && now.isBefore(authPO.getEndTime()))){
                    throw new BusinessException("您购买的该课程已过期，请重新购买");
                }
            }

        }


    }

    public void favoriteCourse(Long courseId) {
        Long userId = UserContext.getUserId();
        UserFavoritesPO po = userFavoritesService.findByUserIdAndCourseId(userId,courseId);
        if (po == null){
            po = new UserFavoritesPO();
            po.setUserId(userId);
            po.setCourseId(courseId);
            userFavoritesService.save(po);
        }
    }

    public void removeFavoriteCourse(Long courseId) {
        Long userId = UserContext.getUserId();
        userFavoritesService.removeByUserIdAndCourseId(userId,courseId);
    }

    public PageResult<SearchCourseResultModel> myFavoriteQueryPageList(QueryFavoriteCourseModel model) {
        model.setUserId(UserContext.getUserId());
        return userFavoritesService.queryPageList(model);
    }

    public void removeFavorite(List<Long> courseIds) {
        Long userId = UserContext.getUserId();
        for (Long courseId : courseIds){
            userFavoritesService.removeByUserIdAndCourseId(userId,courseId);
        }
    }
}
