package com.qkdata.common.aspect;

import com.qkdata.common.annotation.ApiLimit;
import com.qkdata.common.base.exception.BusinessException;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.ValueOperations;
import org.springframework.stereotype.Component;

import java.lang.reflect.Method;
import java.util.concurrent.TimeUnit;

@Aspect
@Component
@Slf4j
public class ApiLimitAspect {
    @Autowired
    private RedisTemplate<String,Object> redisTemplate;

    @Pointcut("@annotation(com.qkdata.common.annotation.ApiLimit)")
    public void logPointCut() {

    }

    @Around("logPointCut()")
    public Object around(ProceedingJoinPoint point) throws Throwable {
        ApiLimit apiLimit = getAnnotation(point);
        return dealLimit(point,apiLimit,false);
    }
    private Object dealLimit(ProceedingJoinPoint point,ApiLimit apiLimit,boolean flag) throws Throwable {
        String msgKey = getMsgKey(point);
        ValueOperations<String, Object> valueOperations = redisTemplate.opsForValue();
        long methodCounts = valueOperations.increment(msgKey, 1);

        // 如果该key不存在，则从0开始计算，并且当count为1的时候，设置过期时间
        if (methodCounts == 1) {
            redisTemplate.expire(msgKey, apiLimit.timeSecond(), TimeUnit.SECONDS);
        }
        // 如果redis中的count大于限制的次数，则等待10秒重试
        if (methodCounts > apiLimit.limitCounts()) {
            if (!flag) {
                //等待10秒后，第一次重试
                Thread.sleep(10 * 1000);
                log.warn("等待10秒后，第一次重试...");

                // 递归，再次请求业务方法
                return dealLimit(point, apiLimit,true);
            } else {
                //如果第一次请求被限制了，等待10秒后重试，如果再次失败，则抛出异常
                throw new BusinessException("请求超时");
            }
        }else {
            return point.proceed();
        }
    }
    private ApiLimit getAnnotation(JoinPoint joinPoint) {
        Signature signature = joinPoint.getSignature();
        MethodSignature methodSignature = (MethodSignature) signature;
        Method method = methodSignature.getMethod();
        return method.getAnnotation(ApiLimit.class);
    }
    private String getMsgKey(ProceedingJoinPoint point){
        MethodSignature signature = (MethodSignature) point.getSignature();
        String className = point.getTarget().getClass().getName();
        String methodName = signature.getName();
        String msgKey = className + "." + methodName;
        return msgKey;
    }
}
