package com.qkdata.biz.sys.service;

import cn.hutool.core.date.DateUtil;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Strings;
import com.google.common.collect.Maps;
import com.qkdata.biz.common.BizConstants;
import com.qkdata.biz.enums.AccountStatusEnum;
import com.qkdata.biz.management.entity.OrganizationPO;
import com.qkdata.biz.management.service.OrganizationService;
import com.qkdata.biz.sys.entity.SysMenuPO;
import com.qkdata.biz.sys.entity.SysUserPO;
import com.qkdata.biz.sys.vo.LoginUserInfo;
import com.qkdata.biz.sys.vo.SysRoleModel;
import com.qkdata.biz.web.vo.FullUserInfo;
import com.qkdata.biz.web.vo.LoginByCodeModel;
import com.qkdata.common.base.exception.BusinessException;
import com.qkdata.common.jwt.JWTService;
import com.qkdata.common.oauth.AuthorizedUser;
import lombok.extern.slf4j.Slf4j;
import org.apache.shiro.crypto.hash.Sha256Hash;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Service
public class ShiroService {
    @Autowired
    private SysUserService sysUserService;
    @Autowired
    private SysMenuService sysMenuService;
    @Autowired
    private ObjectMapper objectMapper;
    @Autowired
    private JWTService jwtService;
    @Autowired
    private SysRoleService sysRoleService;
    @Autowired
    private OrganizationService orgSerivce;
    @Autowired
    @Qualifier("stringRedisTemplate")
    private StringRedisTemplate redisTemplate;
    @Value("${dev.test}")
    private boolean devTest;


    public Set<String> getUserPermissions(Long userId) {
        List<String> permsList;

        //系统管理员，拥有最高权限
        if(userId == 1){
            List<SysMenuPO> menuList = sysMenuService.list();
            permsList = new ArrayList<>(menuList.size());
            for(SysMenuPO menu : menuList){
                permsList.add(menu.getPerms());
            }
        }else{
            permsList = sysUserService.queryAllPerms(userId);
        }
        //用户权限列表
        Set<String> permsSet = new HashSet<>();
        for(String perms : permsList){
            if(StringUtils.isEmpty(perms)){
                continue;
            }
            permsSet.addAll(Arrays.asList(perms.trim().split(",")));
        }
        return permsSet;
    }

    public LoginUserInfo login(String username, String password) throws JsonProcessingException {
        SysUserPO userPO = getUserByUserName(username);
        if(userPO == null || !userPO.getPassword().equals(new Sha256Hash(password, userPO.getSalt()).toHex())) {
            throw new BusinessException("用户名或密码错误");
        }
        if (userPO.getStatus() == AccountStatusEnum.DISABLE){
            throw new BusinessException("帐号已禁用");
        }
        if (userPO.getStatus() == AccountStatusEnum.UNACTIVATE){
            userPO.setActivateTime(LocalDateTime.now());
            sysUserService.updateById(userPO);
        }
        String token = generatorToken(userPO.getId(),userPO.getUsername());
        LoginUserInfo loginUser = new LoginUserInfo();
        BeanUtils.copyProperties(userPO,loginUser);
        loginUser.setAuthorization(token);
        loginUser.setRoles(sysRoleService.getUserRoles(userPO.getId()));
        return loginUser;
    }

    private String generatorToken(Long id,String username) throws JsonProcessingException {
        AuthorizedUser user = new AuthorizedUser();
        user.setUserId(id);
        user.setUsername(username);
        String userJson = objectMapper.writeValueAsString(user);
        Map<String,Object> userClaim = Maps.newConcurrentMap();
        userClaim.put("user",userJson);
        return jwtService.createJWT(userClaim);
    }

    public SysUserPO getUserByUserName(String username) {
        return sysUserService.getByUsername(username);
    }

    public Set<String> getUserRoles(Long userId) {
        List<SysRoleModel> roles = sysRoleService.getUserRoles(userId);
        return roles.stream().map(SysRoleModel::getCode).collect(Collectors.toSet());
    }

    public FullUserInfo loginByCode(LoginByCodeModel model) {
        if (!devTest){
            String cache_captcha = (String) redisTemplate.opsForValue().get(BizConstants.MOBILE_CAPTCHA_PREFIX+model.getMobile());
            if (!Strings.isNullOrEmpty(cache_captcha)){
                cache_captcha = cache_captcha.trim();
                if (!cache_captcha.equals(model.getCode())){
                    log.error("验证码错误：redis值为{}，输入值为{}",cache_captcha,model.getCode());
                    throw new BusinessException("验证码错误");
                }
            }else {
                throw new BusinessException("验证码错误");
            }
        }

        FullUserInfo userInfo = sysUserService.findFullUserInfo(model.getMobile());
        if (userInfo == null){
            //用户不存在时需要注册
            registerUser(model.getMobile());
            userInfo = sysUserService.findFullUserInfo(model.getMobile());
        }
        if (userInfo.getStatus() == AccountStatusEnum.DISABLE){
            throw new BusinessException("对不起，您的帐户已禁用，请联系客服人员");
        }
        if (userInfo.getStatus() == AccountStatusEnum.UNACTIVATE){
            //激活用户
            SysUserPO userPO = new SysUserPO();
            userPO.setId(userInfo.getId());
            userPO.setStatus(AccountStatusEnum.ENABLE);
            userPO.setActivateTime(LocalDateTime.now());
            sysUserService.updateById(userPO);
        }
        //判断帐户角色是否为企业管理员或普通用户
        List<SysRoleModel> roleModels = sysRoleService.getUserRoles(userInfo.getId());
        boolean hasPerm = false;
        for (SysRoleModel role : roleModels){
            if (role.getCode().equals(BizConstants.ROLE_ENTERPRISE_ADMIN)){
                hasPerm = true;
                userInfo.setEnterpriesMgr(true);
            }else if (role.getCode().equals(BizConstants.ROLE_USER)){
                hasPerm = true;
                userInfo.setEnterpriesMgr(false);
            }
        }
        if (!hasPerm){
            throw new BusinessException("对不起，您没有权限登陆，请联系客服人员");
        }
        //获取所属企业
        if (userInfo.getEnterpriseId() != null){
            OrganizationPO orgPo = orgSerivce.getById(userInfo.getEnterpriseId());
            if (orgPo != null){
                userInfo.setEnterpriseName(orgPo.getName());
            }
        }
        updateLastLoginTime(userInfo.getId());
        try {
            userInfo.setAuthorization(generatorToken(userInfo.getId(),userInfo.getUsername()));
        } catch (JsonProcessingException e) {
            throw new BusinessException("系统处理错误");
        }
        return userInfo;
    }

    private void updateLastLoginTime(Long id) {
        SysUserPO po = new SysUserPO();
        po.setLastLoginTime(LocalDateTime.now());
        po.setId(id);
        sysUserService.updateById(po);
    }


    private void registerUser(String mobile) {
        sysUserService.registerUser(mobile);

    }
}
