package com.qkdata.biz.management.service;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import com.aliyuncs.exceptions.ClientException;
import com.aliyuncs.vod.model.v20170321.GetVideoInfoResponse;
import com.aliyuncs.vod.model.v20170321.GetVideoPlayAuthResponse;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.google.common.collect.Lists;
import com.qkdata.biz.aliyun.service.AliyunService;
import com.qkdata.biz.enums.CourseAllowEnum;
import com.qkdata.biz.enums.CourseChargeModelEnum;
import com.qkdata.biz.enums.CourseModelEnum;
import com.qkdata.biz.enums.CourseStatusEnum;
import com.qkdata.biz.management.entity.*;
import com.qkdata.biz.management.mapper.*;
import com.qkdata.biz.management.vo.*;
import com.qkdata.biz.sys.entity.SysUserPO;
import com.qkdata.biz.sys.service.SysUserService;
import com.qkdata.biz.web.vo.FullUserInfo;
import com.qkdata.biz.web.vo.SearchCourseConditionModel;
import com.qkdata.biz.web.vo.SearchCourseResultModel;
import com.qkdata.common.base.enums.CodeEnum;
import com.qkdata.common.base.exception.BusinessException;
import com.qkdata.common.base.model.PageResult;
import com.qkdata.common.util.UserContext;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.stream.Collectors;

/**
 * <p>
 * 课程管理 服务类
 * </p>
 *
 * @author liuyang
 * @since 2021-05-19
 */
@Service
public class CourseService extends ServiceImpl<CourseMapper, CoursePO> {
    @Autowired
    private AllowOrgCourseMapper allowOrgCourseMapper;
    @Autowired
    private CourseChapterMapper chapterMapper;
    @Autowired
    private CourseSeriesMapper seriesMapper;
    @Autowired
    private CourseTagRelMapper tagRelMapper;
    @Autowired
    private CourseAttachmentMapper attachmentMapper;
    @Autowired
    private CourseTagService tagService;
    @Autowired
    private TeacherService teacherService;
    @Autowired
    private AliyunService aliyunService;
    @Autowired
    private ResourceService resourceService;
    @Autowired
    private SysUserService sysUserService;
    @Autowired
    private UserFavoritesService userFavoritesService;
    @Autowired
    private RecommendConfigService recommendConfigService;

    public PageResult<CourseListItemModel> queryPageList(QueryCourseModel param) {
        Page page = new Page(param.getPageIndex(), param.getPageSize());
        List<CourseListItemModel> list = baseMapper.queryPageList(page, param);
        for (CourseListItemModel course : list) {
            List<CourseTagsModel> tagsList = tagService.findListByCourseId(course.getId());
            if (CollUtil.isNotEmpty(tagsList)) {
                List<String> tagNameList = tagsList.stream().map(CourseTagsModel::getTagName).collect(Collectors.toList());
                course.setTags(tagNameList);
            }
        }
        return PageResult.<CourseListItemModel>builder().code(CodeEnum.SUCCESS.getCode()).count(page.getTotal()).data(list).build();
    }



    public CourseDetailModel getCourseDetail(Long id) {
        CoursePO po = getById(id);
        if (po == null) {
            throw new BusinessException("请求错误，课程不存在");
        }
        CourseDetailModel model = new CourseDetailModel();
        BeanUtils.copyProperties(po, model);
        addSeriesInfo(model);
        addTeacherInfo(model);
        addTags(model);
        addChaperInfo(model);
        addAttachmentInfo(model);
        addOrgInfo(model);
        setFavorite(model);
        return model;
    }

    private void setFavorite(CourseDetailModel model) {
        UserFavoritesPO po = userFavoritesService.findByUserIdAndCourseId(UserContext.getUserId(),model.getId());
        if (po != null){
            model.setFavorite(true);
        }
    }

    private void addOrgInfo(CourseDetailModel model) {
        if (model.getAllow() == CourseAllowEnum.SPECIFY){
            List<CourseAllowEnterpriseModel> allowOrgList = baseMapper.selectOrgListByCourseId(model.getId());
            model.setAllowOrgList(allowOrgList);
        }
    }

    private void addAttachmentInfo(CourseDetailModel model) {
        List<CourseAttachmentModel> attachmentModelList = baseMapper.selectAttachmentListByCourseId(model.getId());
        model.setAttachmentList(attachmentModelList);
    }

    private void addChaperInfo(CourseDetailModel model) {
        List<CourseChaperModel> chaperList = chaperList(model.getId());
        model.setChaperList(chaperList);
    }

    private void addTags(CourseDetailModel model) {
        List<CourseTagsModel> tags = tagRelMapper.selectModelList(model.getId());
        model.setTagList(tags);
    }

    private void addSeriesInfo(CourseDetailModel model) {
        CourseSeriesPO po = seriesMapper.selectById(model.getSeriesId());
        if (po != null) {
            model.setSeriesName(po.getName());
        }
    }

    private void addTeacherInfo(CourseDetailModel model) {
        TeacherPO teacherPO = teacherService.getById(model.getTeacherId());
        if (teacherPO != null) {
            model.setTeacherName(teacherPO.getName());
            model.setTeacherAvatar(teacherPO.getAvatarUrl());
            model.setTeacherIntroduce(teacherPO.getIntroduce());
        }
    }

    public void modifyChaperName(ModifyChaperNameModel model) {
        CourseChapterPO po = chapterMapper.selectById(model.getChaperId());
        if (po == null) {
            throw new BusinessException("请求错误，章节不存在");
        }
        po.setName(model.getChaperName());
        chapterMapper.updateById(po);
    }

    @Transactional
    public void delChaper(Long chaperId) {
        CourseChapterPO po = chapterMapper.selectById(chaperId);
        if (po != null) {
            chapterMapper.deleteById(chaperId);
            //删除资源
            ResourcePO resourcePO = resourceService.getById(po.getResourceId());
            if (resourcePO != null){
                resourceService.removeById(po.getResourceId());
            }
            if (StrUtil.isNotBlank(resourcePO.getVideoId())){
                try {
                    aliyunService.removeVideo(resourcePO.getVideoId());
                } catch (ClientException e) {
                    log.error(e.getMessage(),e);
                }
            }

        }


    }

    public void modifyStatus(ModifyCourseStatusModel model) {
        baseMapper.updateStatusById(model.getId(), model.getStatus());
    }

    public List<CourseChaperModel> chaperList(Long courseId) {
        List<CourseChaperModel> list = Lists.newArrayList();
        List<CourseChapterPO> poList = chapterMapper.selectList(Wrappers.<CourseChapterPO>lambdaQuery().eq(CourseChapterPO::getCourseId, courseId).orderByAsc(CourseChapterPO::getSortNo));
        if (CollUtil.isNotEmpty(poList)) {
            for (CourseChapterPO po : poList) {
                CourseChaperModel model = new CourseChaperModel();
                model.setId(po.getId());
                model.setName(po.getName());
                model.setCourseId(po.getCourseId());
                model.setSortNo(po.getSortNo());
                Long resourceId = po.getResourceId();
                setVideoInfo(resourceId, model);
                list.add(model);
            }
        }
        return list;
    }

    private void setVideoInfo(Long resourceId, CourseChaperModel model) {
        ResourcePO resourcePO = resourceService.getById(resourceId);
        if (resourcePO != null) {
            String videoId = resourcePO.getVideoId();
            model.setResourceId(resourceId);
            model.setDuration(resourcePO.getVideoDuration());
            model.setSize(resourcePO.getSize());

            try {
                GetVideoInfoResponse response = aliyunService.getVideoInfo(videoId);
                GetVideoInfoResponse.Video videoInfo = response.getVideo();
                if (videoInfo != null) {
                    if (StrUtil.isNotBlank(videoInfo.getCoverURL())){
                        model.setCoverUrl(videoInfo.getCoverURL());

                    }

                }
            } catch (ClientException e) {
                log.error(e.getMessage(), e);
            }


        }
    }



    public PageResult<CourseSeriesPO> querySeriesPageList(QueryCourseSeriesModel model) {
        Page page = new Page(model.getPageIndex(), model.getPageSize());
        List<CourseSeriesPO> list = seriesMapper.queryPageList(page, model);
        return PageResult.<CourseSeriesPO>builder().code(CodeEnum.SUCCESS.getCode()).count(page.getTotal()).data(list).build();
    }

    public Long saveSeries(CourseSeriesPO po) {
        seriesMapper.insert(po);
        return po.getId();
    }

    @Transactional
    public void saveCourse(SaveCourseModel model) {
        Long id = saveOrUpdateCourseBaseInfo(model);
        saveCourseChaper(id, model.getChaperList());
        saveCourseAttachment(id, model.getAttachmenIds());
        saveOrUpdateCourseTags(id, model.getTagIds());
        if (model.getAllow() == CourseAllowEnum.SPECIFY){
            saveOrUpdateOrg(id,model.getAllowOrgIds());
        }

    }

    private void saveOrUpdateOrg(Long courseId, List<Long> allowOrgIds) {
        if (CollUtil.isNotEmpty(allowOrgIds)){
            allowOrgCourseMapper.deleteByCourseId(courseId);
            for (Long orgId : allowOrgIds){
                AllowOrgCoursePO po = new AllowOrgCoursePO();
                po.setCourseId(courseId);
                po.setOrgId(orgId);
                allowOrgCourseMapper.insert(po);
            }
        }
    }

    private void saveOrUpdateCourseTags(Long courseId, List<Long> tagIds) {
        if (CollUtil.isNotEmpty(tagIds)){
            tagRelMapper.deleteByCourseId(courseId);
            for (Long tagId : tagIds){
                CourseTagRelPO po = new CourseTagRelPO();
                po.setCourseId(courseId);
                po.setTagId(tagId);
                tagRelMapper.insert(po);
            }
        }
    }

    private void saveCourseAttachment(Long courseId, List<Long> attachmenIds) {
        if (CollUtil.isNotEmpty(attachmenIds)){
            attachmentMapper.deleteByCourseId(courseId);
            for (Long attachmentId : attachmenIds){
                CourseAttachmentPO po = new CourseAttachmentPO();
                po.setCourseId(courseId);
                po.setResourceId(attachmentId);
                attachmentMapper.insert(po);
            }
        }
    }

    private void saveCourseChaper(Long courseId, List<SaveCourseChaperModel> chaperList) {
        if (CollUtil.isNotEmpty(chaperList)){
            for (SaveCourseChaperModel chaperModel : chaperList){
                Long chaperId = chaperModel.getId();
                if (chaperId == null){
                    //只做章节新增保存
                    ResourcePO resourcePO = new ResourcePO();
                    resourcePO.setName(chaperModel.getName());
                    resourcePO.setVideoId(chaperModel.getVideoId());
                    try {
                        GetVideoInfoResponse videoInfoResponse = aliyunService.getVideoInfo(chaperModel.getVideoId());
                        resourcePO.setSize(videoInfoResponse.getVideo().getSize());
                        resourcePO.setVideoDuration(videoInfoResponse.getVideo().getDuration());
                    } catch (ClientException e) {
                        log.error(e.getMessage(),e);
                    }
                    resourceService.save(resourcePO);
                    CourseChapterPO chapterPO = new CourseChapterPO();
                    chapterPO.setCourseId(courseId);
                    chapterPO.setSortNo(chaperModel.getSortNo());
                    chapterPO.setName(chaperModel.getName());
                    chapterPO.setResourceId(resourcePO.getId());
                    chapterMapper.insert(chapterPO);
                }
            }
        }
    }

    private Long saveOrUpdateCourseBaseInfo(SaveCourseModel model) {
        CoursePO po = new CoursePO();
        po.setId(model.getId());
        po.setName(model.getName());
        po.setType(model.getType());
        po.setIntroduce(model.getIntroduce());
        po.setDetail(model.getDetail());
        po.setLogoUrl(model.getLogoUrl());
        po.setStatus(CourseStatusEnum.UP);
        po.setTeacherId(model.getTeacherId());
        po.setChargeModel(model.getChargeModel());
        if (po.getChargeModel() == CourseChargeModelEnum.PAY) {
            po.setPrice(model.getPrice());
            po.setValidPeriod(model.getValidPeriod());
            po.setVipPrice(model.getVipPrice());
        }
        po.setModel(CourseModelEnum.VOD);
        po.setSeriesId(model.getSeriesId());
        po.setAllow(model.getAllow());
        saveOrUpdate(po);
        return po.getId();
    }

    public VideoInfoModel getVideoInfo(String videoId) throws ClientException {
        VideoInfoModel model = new VideoInfoModel();
        GetVideoInfoResponse response = aliyunService.getVideoInfo(videoId);
        model.setVideoId(videoId);
        model.setSize(response.getVideo().getSize());
        model.setDuration(response.getVideo().getDuration());
        model.setCoverURL(response.getVideo().getCoverURL());
        return model;
    }

    public PageResult<SearchCourseResultModel> search(SearchCourseConditionModel model) {
        Page page = new Page(model.getPageIndex(),model.getPageSize());
        List<SearchCourseResultModel> list = baseMapper.searchPageList(page,model);
        for (SearchCourseResultModel m : list){
            List<CourseTagsModel> tags = tagRelMapper.selectModelList(m.getCourseId());
            m.setTags(tags);
            List<CourseChaperModel> chaperList = chaperList(m.getCourseId());
            if (CollUtil.isNotEmpty(chaperList)){
                m.setDuration(chaperList.get(0).getDuration());
            }
        }
        return PageResult.<SearchCourseResultModel>builder().code(CodeEnum.SUCCESS.getCode()).count(page.getTotal()).data(list).build();
    }

    public List<Long> getAllowEnterpriseList(Long courseId) {
        List<AllowOrgCoursePO> enterpriseIdList = allowOrgCourseMapper.selectList(Wrappers.<AllowOrgCoursePO>lambdaQuery().eq(AllowOrgCoursePO::getCourseId,courseId));
        if (CollUtil.isNotEmpty(enterpriseIdList)){
            return enterpriseIdList.stream().map(AllowOrgCoursePO::getOrgId).collect(Collectors.toList());
        }
        return Lists.newArrayList();
    }

    /**
     * 验证当前登陆用户是否有该课程权限
     * @param courseId
     */
    public void checkPerm(Long courseId) {
        boolean hasPerm = false;
        CoursePO coursePO = getById(courseId);
        if (coursePO == null){
            throw new BusinessException("请求错误，该课程不存在");
        }
        if (coursePO.getStatus() == CourseStatusEnum.DOWN){
            throw new BusinessException("该课程已下架");
        }
        if (coursePO.getAllow() == CourseAllowEnum.SPECIFY){
            SysUserPO sysUserPO = UserContext.getUser();
            if (sysUserPO == null){
                throw new BusinessException("请登陆后查看");
            }
            List<Long> allowOrgIds = getAllowEnterpriseList(courseId);
            if (CollUtil.isNotEmpty(allowOrgIds)){
                String username = UserContext.getUser().getUsername();
                FullUserInfo userInfo = sysUserService.findFullUserInfo(username);
                if (userInfo.getEnterpriseId() != null){
                    for (Long orgId : allowOrgIds){
                        if (orgId.longValue() == userInfo.getEnterpriseId().longValue()){
                            hasPerm = true;
                            break;
                        }
                    }
                }
            }

        }else {
            hasPerm = true;
        }
        if (!hasPerm){
            throw new BusinessException("对不起，您没权限查看该课程");
        }
    }


    public GetVideoPlayAuthResponse getVideoPlayAuth(Long courseId,Long chaperId){
        CourseChapterPO po = chapterMapper.selectById(chaperId);
        if (po != null){
            ResourcePO resourcePO = resourceService.getById(po.getResourceId());
            if (resourcePO != null && StrUtil.isNotBlank(resourcePO.getVideoId())){
                try {
                    GetVideoPlayAuthResponse response = aliyunService.getVideoPlayAuth(resourcePO.getVideoId());
                    return response;
                } catch (ClientException e) {
                    throw new BusinessException("获取视频凭证异常");
                }
            }
        }
        return null;
    }
    private List<CoursePO> findListByTags(List<Long> tagIds){
        return baseMapper.findListByTags(tagIds);
    }
    /**
     * 根据课程标签关联推荐课程列表
     * 默认推荐4个课程
     * 优先推荐相同标签下的其它课程
     * 如果相同标签下没有其它课程，则随机推荐其它课程
     * 注意过滤指定企业可见范围的课程
     * @return
     */
    public List<SearchCourseResultModel> relationRecommend(Long courseId) {
        CoursePO coursePO = getById(courseId);
        if (coursePO != null){
            throw new BusinessException("请求错误，课程不存在");
        }
        List<SearchCourseResultModel> resultList = Lists.newArrayList();
        List<CourseTagsModel> courseTagsModelList = tagService.findListByCourseId(courseId);
        List<Long> tagIds = courseTagsModelList.stream().map(CourseTagsModel::getTagId).collect(Collectors.toList());
        List<CoursePO> courseList = findListByTags(tagIds);
        int index = 0;
        for (CoursePO course : courseList){
            if (index == 3){
                break;
            }else {
                SearchCourseResultModel model = new SearchCourseResultModel();
                model.setCourseId(course.getId());
                model.setCourseName(course.getName());
                model.setCourseLogoUrl(course.getLogoUrl());
                resultList.add(model);
                index++;
            }

        }
        if (resultList.size() < 4){
            int diff = 4 - resultList.size();
            List<Long> ids = resultList.stream().map(SearchCourseResultModel::getCourseId).collect(Collectors.toList());
            courseList = list(Wrappers.<CoursePO>lambdaQuery()
                    .notIn(CoursePO::getId,ids)
                    .eq(CoursePO::getStatus,CourseStatusEnum.UP)
                    .eq(CoursePO::getAllow,CourseAllowEnum.ALL)
                    .last("limit "+diff));
            for (CoursePO course : courseList){
                SearchCourseResultModel model = new SearchCourseResultModel();
                model.setCourseId(course.getId());
                model.setCourseName(course.getName());
                model.setCourseLogoUrl(course.getLogoUrl());
                resultList.add(model);
            }
        }
        for (SearchCourseResultModel model : resultList){
            List<CourseTagsModel> tagList = tagService.findListByCourseId(model.getCourseId());
            model.setTags(tagList);
            List<CourseChaperModel> chaperList = chaperList(model.getCourseId());
            if (CollUtil.isNotEmpty(chaperList)){
                CourseChaperModel chaperModel = chaperList.get(0);
                ResourcePO resourcePO = resourceService.getById(chaperModel.getResourceId());
                if (resourcePO != null){
                    model.setDuration(resourcePO.getVideoDuration());

                }
            }
        }

        return resultList;
    }

    public PageResult<SearchCourseResultModel> mainPageRecommendList(SearchCourseConditionModel model) {
        Page page = new Page(model.getPageIndex(),model.getPageSize());
        List<SearchCourseResultModel> list = baseMapper.recommendList(page,model);
        for (SearchCourseResultModel recommend : list){
            List<CourseTagsModel> tagList = tagService.findListByCourseId(recommend.getCourseId());
            recommend.setTags(tagList);
            List<CourseChaperModel> chaperList = chaperList(recommend.getCourseId());
            if (CollUtil.isNotEmpty(chaperList)){
                CourseChaperModel chaperModel = chaperList.get(0);
                ResourcePO resourcePO = resourceService.getById(chaperModel.getResourceId());
                if (resourcePO != null){
                    recommend.setDuration(resourcePO.getVideoDuration());

                }
            }
        }
        return PageResult.<SearchCourseResultModel>builder().code(CodeEnum.SUCCESS.getCode()).count(page.getTotal()).data(list).build();
    }
}