package com.qkdata.biz.sys.controller;


import com.qkdata.biz.common.BizConstants;
import com.qkdata.biz.sys.entity.SysUserPO;
import com.qkdata.biz.sys.service.SysUserService;
import com.qkdata.biz.sys.vo.PasswordModel;
import com.qkdata.biz.sys.vo.QueryUserModel;
import com.qkdata.biz.sys.vo.SysUserListModel;
import com.qkdata.biz.sys.vo.SysUserModel;
import com.qkdata.common.annotation.SysLog;
import com.qkdata.common.base.exception.BusinessException;
import com.qkdata.common.base.model.PageResult;
import com.qkdata.common.base.model.Result;
import com.qkdata.common.constants.AddGroup;
import com.qkdata.common.constants.UpdateGroup;
import com.qkdata.common.util.UserContext;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.shiro.authz.annotation.Logical;
import org.apache.shiro.authz.annotation.RequiresRoles;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.Arrays;

@Api(tags = "系统用户管理")
@RestController
@RequestMapping("/api/sys/user")
public class SysUserController {
    @Autowired
    private SysUserService sysUserService;

    @ApiOperation("查询用户列表")
    @PostMapping("/list")
    @RequiresRoles(value = {BizConstants.ROLE_ADMIN,BizConstants.ROLE_OPERATOR},logical = Logical.OR)
    public PageResult<SysUserListModel> list(@RequestBody QueryUserModel queryUserModel){
        return sysUserService.queryPageList(queryUserModel);
    }

    @ApiOperation("获取当前用户信息")
    @GetMapping("/info")
    @RequiresRoles(value = {BizConstants.ROLE_ADMIN,BizConstants.ROLE_OPERATOR},logical = Logical.OR)
    public Result<SysUserModel> info(){
        SysUserPO userPO =  UserContext.getUser();
        SysUserModel dto = new SysUserModel();
        BeanUtils.copyProperties(userPO,dto);
        dto.setRoleIdList(sysUserService.queryRoleIdList(userPO.getId()));
        return Result.succeed(dto);
    }
    @ApiOperation("获取某个用户信息")
    @GetMapping("/info/{id}")
    @RequiresRoles(value = {BizConstants.ROLE_ADMIN})
    public Result<SysUserModel> infoById(@PathVariable Long id){
        SysUserPO sysUserPO = sysUserService.getById(id);
        SysUserModel dto = new SysUserModel();
        BeanUtils.copyProperties(sysUserPO,dto);
        dto.setRoleIdList(sysUserService.queryRoleIdList(id));
        return Result.succeed(dto);
    }
    @ApiOperation("保存用户信息")
    @SysLog("保存用户信息")
    @PostMapping("/save")
    @RequiresRoles(value = {BizConstants.ROLE_ADMIN})
    public Result<String> save(@RequestBody @Validated(AddGroup.class) SysUserModel sysUserModel){
        sysUserService.saveUser(sysUserModel);
        return Result.succeed("ok");
    }
    @ApiOperation("修改用户信息")
    @SysLog("修改用户信息")
    @PostMapping("/update")
    @RequiresRoles(value = {BizConstants.ROLE_ADMIN})
    public Result<String> update(@RequestBody @Validated(UpdateGroup.class) SysUserModel sysUserModel){
        sysUserService.updateUser(sysUserModel);
        return Result.succeed("ok");
    }

    /**
     * 修改登录用户密码
     */
    @SysLog("修改当前用户密码")
    @ApiOperation("修改当前用户密码")
    @PostMapping("/password")
    public Result<String> password(@RequestBody @Valid PasswordModel form){
        //更新密码
        sysUserService.updatePassword(UserContext.getUser(), form.getPassword(), form.getNewPassword());
        return Result.succeed("ok");

    }

    /**
     * 删除用户
     */
    @ApiOperation("删除用户")
    @SysLog("删除用户")
    @PostMapping("/delete")
    @RequiresRoles(value = {BizConstants.ROLE_ADMIN})
    public Result<String> delete(@RequestBody Long[] userIds){
        if(ArrayUtils.contains(userIds, 1L)){
            throw new BusinessException("系统管理员不能删除");
        }

        if(ArrayUtils.contains(userIds, UserContext.getUserId())){
            throw new BusinessException("当前用户不能删除");
        }

        sysUserService.removeUsers(Arrays.asList(userIds));
        return Result.succeed("ok");
    }
}
